"use client"

import React from "react"

import { useState, useEffect, useRef } from "react"
import { Button } from "@/components/ui/button"
import { Card } from "@/components/ui/card"
import Link from "next/link"
import {
  Menu,
  X,
  ArrowRight,
  Code,
  Palette,
  Zap,
  Brain,
  Rocket,
  Target,
  Plus,
  Minus,
  Award,
  Users,
  Clock,
  TrendingUp,
} from "lucide-react"
import Image from "next/image"

export default function Portfolio() {
  const [mounted, setMounted] = useState(false)
  const [isMenuOpen, setIsMenuOpen] = useState(false)
  const [activeFilter, setActiveFilter] = useState("all")
  const [openFaq, setOpenFaq] = useState<number | null>(null)
  const [codeIndex, setCodeIndex] = useState(0)
  const [currentTestimonial, setCurrentTestimonial] = useState(0)
  const [testimonialGroups, setTestimonialGroups] = useState<any[][]>([])

  const codeLines = [
    "const portfolio = {",
    "  name: 'Kristijan',",
    "  role: 'Digital Innovator',",
    "  skills: ['Web Design', 'AI Integration'],",
    "  passion: 'Creating digital magic'",
    "}",
  ]

  useEffect(() => {
    setMounted(true)
    const interval = setInterval(() => {
      setCodeIndex((prev) => (prev + 1) % codeLines.length)
    }, 2000)
    return () => clearInterval(interval)
  }, [])

  useEffect(() => {
    const groups = []
    for (let i = 0; i < testimonials.length; i += 3) {
      groups.push(testimonials.slice(i, i + 3))
    }
    setTestimonialGroups(groups)
  }, [])

  useEffect(() => {
    const interval = setInterval(() => {
      setCurrentTestimonial((prev) => (prev + 1) % testimonialGroups.length)
    }, 4000) // Auto-rotate every 4 seconds
    return () => clearInterval(interval)
  }, [testimonialGroups.length])

  const projects = [
    {
      id: 1,
      title: "AI-Powered E-commerce Platform",
      description:
        "Revolutionary online marketplace featuring intelligent product recommendations, dynamic pricing algorithms, and personalized shopping experiences that increased conversion rates by 340%.",
      image:
        "https://images.unsplash.com/photo-1460925895917-afdab827c52f?ixlib=rb-4.0.3&ixid=M3wxMjA3fDB8MHxwaG90by1wYWdlfHx8fGVufDB8fHx8fA%3D%3D&auto=format&fit=crop&w=2070&q=80",
      category: "ecommerce",
      technologies: ["React", "Node.js", "TensorFlow", "Stripe"],
      year: "2024",
      client: "TechCorp Inc.",
    },
    {
      id: 2,
      title: "Smart Banking Application",
      description:
        "Next-generation mobile banking platform with AI-driven fraud detection, biometric authentication, and real-time financial insights that serves over 100K+ active users.",
      image:
        "https://images.unsplash.com/photo-1551288049-bebda4e38f71?ixlib=rb-4.0.3&ixid=M3wxMjA3fDB8MHxwaG90by1wYWdlfHx8fGVufDB8fHx8fA%3D%3D&auto=format&fit=crop&w=2070&q=80",
      category: "applications",
      technologies: ["React Native", "Firebase", "Python", "AWS"],
      year: "2024",
      client: "FinanceFlow",
    },
    {
      id: 3,
      title: "Advanced E-commerce Solution",
      description:
        "Comprehensive multi-vendor marketplace with advanced inventory management, AI-powered customer insights, and automated marketing campaigns.",
      image:
        "https://images.unsplash.com/photo-1557804506-669a67965ba0?ixlib=rb-4.0.3&ixid=M3wxMjA3fDB8MHxwaG90by1wYWdlfHx8fGVufDB8fHx8fA%3D%3D&auto=format&fit=crop&w=2074&q=80",
      category: "ecommerce",
      technologies: ["Vue.js", "Laravel", "MySQL", "Redis"],
      year: "2023",
      client: "MarketPlace Pro",
    },
    {
      id: 4,
      title: "Smart Restaurant Platform",
      description:
        "Intelligent restaurant ecosystem with AI-powered menu optimization, dynamic pricing, real-time analytics, and seamless customer experience management.",
      image:
        "https://images.unsplash.com/photo-1555949963-ff9fe0c870eb?ixlib=rb-4.0.3&ixid=M3wxMjA3fDB8MHxwaG90by1wYWdlfHx8fGVufDB8fHx8fA%3D%3D&auto=format&fit=crop&w=2076&q=80",
      category: "websites",
      technologies: ["Next.js", "PostgreSQL", "OpenAI", "Vercel"],
      year: "2023",
      client: "RestaurantTech",
    },
    {
      id: 5,
      title: "Creative Agency Portfolio",
      description:
        "Stunning portfolio website for a creative agency featuring interactive animations, smooth transitions, and immersive user experience that showcases their creative work.",
      image:
        "https://images.unsplash.com/photo-1561070791-2526d30994b5?ixlib=rb-4.0.3&ixid=M3wxMjA3fDB8MHxwaG90by1wYWdlfHx8fGVufDB8fHx8fA%3D%3D&auto=format&fit=crop&w=2064&q=80",
      category: "websites",
      technologies: ["React", "Framer Motion", "GSAP", "Netlify"],
      year: "2024",
      client: "Creative Studio",
    },
    {
      id: 6,
      title: "Social Media Management App",
      description:
        "Comprehensive social media management platform with AI-powered content scheduling, analytics dashboard, and multi-platform posting capabilities.",
      image:
        "https://images.unsplash.com/photo-1611224923853-80b023f02d71?ixlib=rb-4.0.3&ixid=M3wxMjA3fDB8MHxwaG90by1wYWdlfHx8fGVufDB8fHx8fA%3D%3D&auto=format&fit=crop&w=2039&q=80",
      category: "applications",
      technologies: ["Vue.js", "Node.js", "MongoDB", "Socket.io"],
      year: "2023",
      client: "SocialTech",
    },
    {
      id: 7,
      title: "Brand Identity & Social Media Design",
      description:
        "Complete brand identity design and social media marketing campaign for a tech startup, including logo design, brand guidelines, and social media templates.",
      image:
        "https://images.unsplash.com/photo-1634942537034-2531766767d1?ixlib=rb-4.0.3&ixid=M3wxMjA3fDB8MHxwaG90by1wYWdlfHx8fA%3D%3D&auto=format&fit=crop&w=2070&q=80",
      category: "social-media",
      technologies: ["Adobe Creative Suite", "Figma", "Canva", "After Effects"],
      year: "2024",
      client: "StartupTech",
    },
    {
      id: 8,
      title: "Luxury Fashion E-store",
      description:
        "High-end fashion e-commerce platform with premium user experience, advanced filtering, virtual try-on features, and seamless checkout process.",
      image:
        "https://images.unsplash.com/photo-1441986300917-64674bd600d8?ixlib=rb-4.0.3&ixid=M3wxMjA3fDB8MHxwaG90by1wYWdlfHx8fA%3D%3D&auto=format&fit=crop&w=2070&q=80",
      category: "ecommerce",
      technologies: ["Shopify Plus", "React", "AR.js", "Stripe"],
      year: "2024",
      client: "Luxury Fashion",
    },
    {
      id: 9,
      title: "AI Content Generation Platform",
      description:
        "Revolutionary AI-powered content creation platform that generates high-quality articles, social media posts, and marketing copy using advanced natural language processing and machine learning algorithms.",
      image:
        "https://images.unsplash.com/photo-1677442136019-21780ecad995?ixlib=rb-4.0.3&ixid=M3wxMjA3fDB8MHxwaG90by1wYWdlfHx8fA%3D%3D&auto=format&fit=crop&w=2070&q=80",
      category: "ai-projects",
      technologies: ["OpenAI GPT-4", "Python", "FastAPI", "React"],
      year: "2024",
      client: "ContentAI",
    },
    {
      id: 10,
      title: "Intelligent Document Analysis System",
      description:
        "Advanced AI system that automatically extracts, categorizes, and analyzes information from complex documents, reducing manual processing time by 85% and improving accuracy.",
      image:
        "https://images.unsplash.com/photo-1555949963-aa79dcee981c?ixlib=rb-4.0.3&ixid=M3wxMjA3fDB8MHxwaG90by1wYWdlfHx8fA%3D%3D&auto=format&fit=crop&w=2070&q=80",
      category: "ai-projects",
      technologies: ["TensorFlow", "Computer Vision", "NLP", "Azure"],
      year: "2024",
      client: "DocuTech Solutions",
    },
    {
      id: 11,
      title: "AI-Powered Customer Support Bot",
      description:
        "Intelligent chatbot system with natural language understanding, sentiment analysis, and automated ticket routing that handles 90% of customer inquiries without human intervention.",
      image:
        "https://images.unsplash.com/photo-1531746790731-6c087fecd65a?ixlib=rb-4.0.3&ixid=M3wxMjA3fDB8MHxwaG90by1wYWdlfHx8fA%3D%3D&auto=format&fit=crop&w=2006&q=80",
      category: "ai-projects",
      technologies: ["Dialogflow", "Node.js", "Machine Learning", "WebSocket"],
      year: "2024",
      client: "SupportAI Inc.",
    },
  ]

  const filteredProjects =
    activeFilter === "all" ? projects : projects.filter((project) => project.category === activeFilter)

  const testimonials = [
    {
      name: "Marko Jovanović",
      position: "CEO, TechStart Solutions",
      content:
        "Kristijan's innovative approach to web design and AI integration completely transformed our digital presence. His attention to detail and technical expertise resulted in a 340% increase in conversion rates and significantly improved user engagement.",
      avatar:
        "https://images.unsplash.com/photo-1507003211169-0a1dd7228f2d?ixlib=rb-4.0.3&ixid=M3wxMjA3fDB8MHxwaG90by1wYWdlfHx8fGVufDB8fHx8fA%3D%3D&auto=format&fit=crop&w=687&q=80",
      rating: 5,
      company: "TechStart Solutions",
    },
    {
      name: "Ana Petrović",
      position: "Marketing Director, Bloom Agency",
      content:
        "Working with Kristijan was an absolute game-changer for our agency. His ability to understand complex business requirements and translate them into stunning, functional digital solutions is truly remarkable. The project exceeded all expectations.",
      avatar:
        "https://images.unsplash.com/photo-1494790108755-2616b612b786?ixlib=rb-4.0.3&ixid=M3wxMjA3fDB8MHxwaG90by1wYWdlfHx8fGVufDB8fHx8fA%3D%3D&auto=format&fit=crop&w=1170&q=80",
      rating: 5,
      company: "Bloom Agency",
    },
    {
      name: "Ivan Marković",
      position: "CTO, InnovateTech",
      content:
        "Kristijan's expertise in both cutting-edge web development and AI integration is exceptional. He delivered a sophisticated solution that not only looks fantastic but also provides invaluable business insights through intelligent data analysis.",
      avatar:
        "https://images.unsplash.com/photo-1472099645785-5658abf48d80?ixlib=rb-4.0.3&ixid=M3wxMjA3fDB8MHxwaG90by1wYWdlfHx8fA%3D%3D&auto=format&fit=crop&w=1170&q=80",
      rating: 5,
      company: "InnovateTech",
    },
    {
      name: "Milica Stojanović",
      position: "Founder, Digital Dreams",
      content:
        "The level of professionalism and creativity Kristijan brings to every project is outstanding. He transformed our outdated website into a modern, AI-powered platform that increased our lead generation by 280%. Highly recommended!",
      avatar:
        "https://images.unsplash.com/photo-1544005313-94ddf0286df2?ixlib=rb-4.0.3&ixid=M3wxMjA3fDB8MHxwaG90by1wYWdlfHx8fA%3D%3D&auto=format&fit=crop&w=687&q=80",
      rating: 5,
      company: "Digital Dreams",
    },
    {
      name: "Stefan Nikolić",
      position: "Product Manager, FutureTech",
      content:
        "Kristijan's strategic approach to digital transformation is impressive. He doesn't just build websites; he creates comprehensive digital ecosystems that drive business growth. Our ROI improved by 150% within the first quarter.",
      avatar:
        "https://images.unsplash.com/photo-1500648767791-00dcc994a43e?ixlib=rb-4.0.3&ixid=M3wxMjA3fDB8MHxwaG90by1wYWdlfHx8fA%3D%3D&auto=format&fit=crop&w=687&q=80",
      rating: 5,
      company: "FutureTech",
    },
    {
      name: "Jelena Mitrović",
      position: "Creative Director, Pixel Perfect",
      content:
        "Collaborating with Kristijan was an incredible experience. His blend of technical expertise and creative vision resulted in a website that perfectly captures our brand essence while delivering exceptional user experience and performance.",
      avatar:
        "https://images.unsplash.com/photo-1544005313-94ddf0286df2?ixlib=rb-4.0.3&ixid=M3wxMjA3fDB8MHxwaG90by1wYWdlfHx8fA%3D%3D&auto=format&fit=crop&w=688&q=80",
      rating: 5,
      company: "Pixel Perfect",
    },
    {
      name: "Miloš Petković",
      position: "E-commerce Director, ShopTech",
      content:
        "Kristijan delivered an exceptional e-commerce platform that revolutionized our online business. The AI-powered features and seamless user experience resulted in a 400% increase in sales within six months.",
      avatar:
        "https://images.unsplash.com/photo-1507003211169-0a1dd7228f2d?ixlib=rb-4.0.3&auto=format&fit=crop&w=687&q=80",
      rating: 5,
      company: "ShopTech",
    },
    {
      name: "Sara Živković",
      position: "Marketing Manager, GrowthLab",
      content:
        "Working with Kristijan was transformative for our digital strategy. His innovative approach and attention to detail created a platform that perfectly represents our brand and engages our audience.",
      avatar:
        "https://images.unsplash.com/photo-1494790108755-2616b612b786?ixlib=rb-4.0.3&auto=format&fit=crop&w=1170&q=80",
      rating: 5,
      company: "GrowthLab",
    },
    {
      name: "Nikola Radić",
      position: "Startup Founder, InnovateNow",
      content:
        "Kristijan helped us launch our startup with a cutting-edge platform that impressed investors and users alike. His technical expertise and creative vision exceeded all our expectations.",
      avatar:
        "https://images.unsplash.com/photo-1472099645785-5658abf48d80?ixlib=rb-4.0.3&auto=format&fit=crop&w=1170&q=80",
      rating: 5,
      company: "InnovateNow",
    },
    {
      name: "Tamara Jovanović",
      position: "Digital Director, MediaForce",
      content:
        "The website Kristijan created for us is not just beautiful—it's intelligent. The AI integration has automated many of our processes and significantly improved our operational efficiency.",
      avatar:
        "https://images.unsplash.com/photo-1544005313-94ddf0286df2?ixlib=rb-4.0.3&auto=format&fit=crop&w=687&q=80",
      rating: 5,
      company: "MediaForce",
    },
    {
      name: "Dejan Milanović",
      position: "Tech Lead, DataFlow Solutions",
      content:
        "Kristijan's ability to integrate complex AI systems with intuitive user interfaces is remarkable. He delivered a solution that our technical team and end users both love.",
      avatar:
        "https://images.unsplash.com/photo-1500648767791-00dcc994a43e?ixlib=rb-4.0.3&auto=format&fit=crop&w=687&q=80",
      rating: 5,
      company: "DataFlow Solutions",
    },
    {
      name: "Isidora Stanković",
      position: "Brand Manager, CreativeHub",
      content:
        "From concept to launch, Kristijan demonstrated exceptional professionalism and creativity. Our new platform has become a benchmark in our industry, generating significant business growth.",
      avatar:
        "https://images.unsplash.com/photo-1544005313-94ddf0286df2?ixlib=rb-4.0.3&auto=format&fit=crop&w=688&q=80",
      rating: 5,
      company: "CreativeHub",
    },
  ]

  const faqs = [
    {
      question: "How can AI integration transform my business operations?",
      answer:
        "AI integration can revolutionize your business through intelligent automation, predictive analytics, personalized user experiences, and data-driven decision making. I implement solutions like chatbots for 24/7 customer support, recommendation engines that boost sales by up to 35%, and predictive models that optimize inventory and reduce costs.",
    },
    {
      question: "What makes your design approach unique in today's market?",
      answer:
        "My approach combines cutting-edge design principles with advanced technological innovation, particularly AI integration. I focus on creating not just visually stunning interfaces, but intelligent digital ecosystems that adapt to user behavior, provide personalized experiences, and deliver measurable ROI through data-driven insights.",
    },
    {
      question: "What's your typical project timeline and process?",
      answer:
        "My process involves 5 key phases: Discovery & Strategy (1-2 weeks), Design & Prototyping (2-3 weeks), Development & Integration (4-8 weeks), Testing & Optimization (1-2 weeks), and Launch & Support (ongoing). Complex AI integrations may require 10-16 weeks, but I provide detailed milestones and regular updates throughout.",
    },
    {
      question: "Do you provide long-term support and maintenance?",
      answer:
        "Absolutely. I offer comprehensive support packages including regular security updates, performance optimization, feature enhancements, AI model retraining, and 24/7 monitoring. I believe in building lasting partnerships and ensuring your digital solution continues to evolve with your business needs.",
    },
  ]

  const FAQItem = ({
    faq,
    index,
    isOpen,
    onToggle,
  }: { faq: any; index: number; isOpen: boolean; onToggle: () => void }) => {
    return (
      <div className="group mb-8">
        <div className="bg-white rounded-2xl border border-gray-200 shadow-xl hover:shadow-2xl transition-all duration-300 overflow-hidden">
          <button
            onClick={onToggle}
            className={`w-full p-8 text-left transition-all duration-300 hover:bg-gray-50 flex items-center justify-between group/btn ${
              isOpen ? "bg-gray-50" : ""
            }`}
          >
            <h3 className="text-xl font-semibold text-gray-900 pr-4 group-hover/btn:text-gray-700 transition-colors">
              {faq.question}
            </h3>
            <div className="flex-shrink-0">
              <div className="w-10 h-10 rounded-full bg-gray-100 flex items-center justify-center transition-all duration-300 group-hover/btn:bg-gray-200 group-hover/btn:scale-110 shadow-lg">
                {isOpen ? (
                  <Minus className="w-5 h-5 text-gray-600" />
                ) : (
                  <Plus className="w-5 h-5 text-gray-600" />
                )}
              </div>
            </div>
          </button>
          {isOpen && (
            <div className="px-8 pb-8 animate-in slide-in-from-top-2 duration-300">
              <div className="bg-gradient-to-r from-gray-50 to-gray-100/80 p-8 rounded-xl border border-gray-200/50 shadow-inner">
                <p className="text-gray-700 leading-relaxed text-lg">{faq.answer}</p>
              </div>
            </div>
          )}
        </div>
      </div>
    )
  }

  const nextTestimonial = () => {
    setCurrentTestimonial((prev) => (prev + 1) % testimonialGroups.length)
  }

  const prevTestimonial = () => {
    setCurrentTestimonial((prev) => (prev - 1 + testimonialGroups.length) % testimonialGroups.length)
  }

  if (!mounted) return null

  return (
    <div className="min-h-screen bg-white text-gray-900">
      <header className="sticky top-0 z-50 backdrop-blur-3xl bg-[#161d25]/95 border-b border-white/10 shadow-2xl">
        <div className="container mx-auto px-4 py-6 relative">
          <div className="flex items-center justify-between">
            <div className="flex items-center gap-4">
              <div className="relative group">
                <Image
                  src="https://i.ibb.co/5XLC8VTC/kdsgnpng2.png"
                  alt="Kristijan Digital"
                  width={40}
                  height={40}
                  className="rounded-2xl shadow-lg transition-transform duration-700 group-hover:scale-105 relative z-10"
                />
              </div>
              <div>
                <h1 className="text-2xl font-bold text-white">kristijan.digital</h1>
                <p className="text-xs text-white/60 font-medium">Digital Innovation Specialist</p>
              </div>
            </div>

            {/* Desktop Navigation */}
            <nav className="hidden lg:flex items-center gap-12">
              {["Home", "About", "Expertise", "Projects", "Testimonials"].map((item) => (
                <a
                  key={item}
                  href={`#${item.toLowerCase()}`}
                  className="text-sm font-medium text-white/70 hover:text-white hover:scale-110 transition-all duration-500 relative group"
                >
                  {item}
                  <div className="absolute -bottom-2 left-0 w-0 h-0.5 bg-gradient-to-r from-gray-400 to-gray-300 group-hover:w-full transition-all duration-500"></div>
                </a>
              ))}
            </nav>

            <div className="flex items-center gap-6">
              <Link href="/contact">
                <Button className="hidden lg:flex bg-gradient-to-r from-white/15 to-white/10 backdrop-blur-xl border border-white/30 hover:bg-gradient-to-r hover:from-white/25 hover:to-white/20 hover:border-white/50 shadow-xl hover:shadow-2xl hover:shadow-white/20 transition-all duration-500 hover:scale-105 rounded-xl px-8 py-3 font-semibold text-white hover:glow">
                  Let's Work Together
                  <ArrowRight className="ml-2 h-4 w-4" />
                </Button>
              </Link>

              {/* Mobile Menu Button */}
              <Button
                variant="ghost"
                size="icon"
                className="lg:hidden rounded-2xl bg-white/5 backdrop-blur-xl border border-white/10 hover:bg-white/10 hover:border-white/30 transition-all duration-500"
                onClick={() => setIsMenuOpen(!isMenuOpen)}
              >
                {isMenuOpen ? <X className="h-5 w-5 text-white" /> : <Menu className="h-5 w-5 text-white" />}
              </Button>
            </div>
          </div>

          {/* Mobile Navigation */}
          {isMenuOpen && (
            <nav className="lg:hidden mt-6 pb-6 border-t border-white/10 pt-6 bg-black/20 backdrop-blur-3xl rounded-3xl border border-white/10 shadow-2xl">
              <div className="flex flex-col gap-6">
                {["Home", "About", "Expertise", "Projects", "Testimonials"].map((item) => (
                  <a
                    key={item}
                    href={`#${item.toLowerCase()}`}
                    className="text-sm font-medium text-white/80 hover:text-white transition-colors duration-300 px-4"
                  >
                    {item}
                  </a>
                ))}
                <Link href="/contact" className="px-4">
                  <Button className="w-full mt-4 bg-white/10 backdrop-blur-xl border border-white/20 hover:bg-white/20 rounded-2xl text-white">
                    Let's Collaborate
                  </Button>
                </Link>
              </div>
            </nav>
          )}
        </div>
      </header>

      <section id="home" className="relative py-32 md:py-48 overflow-hidden bg-[#161d25]">
        <div className="absolute inset-0 overflow-hidden">
          <div className="absolute top-[15%] left-[20%] w-2 h-2 bg-white/25 rounded-full animate-float"></div>
          <div className="absolute top-[25%] right-[30%] w-1 h-1 bg-white/35 rounded-full animate-float-delayed"></div>
          <div className="absolute bottom-[20%] left-[35%] w-3 h-3 bg-white/15 rounded-full animate-float-slow"></div>
          <div className="absolute top-[60%] right-[25%] w-1.5 h-1.5 bg-white/30 rounded-full animate-float"></div>
          <div className="absolute bottom-[35%] right-[65%] w-2 h-2 bg-white/20 rounded-full animate-float-delayed"></div>
          <div className="absolute top-[45%] left-[15%] w-1 h-1 bg-white/25 rounded-full animate-float-slow"></div>
          <div className="absolute top-[80%] left-[70%] w-2.5 h-2.5 bg-white/18 rounded-full animate-float"></div>
          <div className="absolute top-[10%] right-[15%] w-1 h-1 bg-white/28 rounded-full animate-float-delayed"></div>
          <div className="absolute bottom-[60%] left-[80%] w-1.5 h-1.5 bg-white/22 rounded-full animate-float-slow"></div>
          <div className="absolute top-[35%] left-[60%] w-1 h-1 bg-white/32 rounded-full animate-float"></div>
          <div className="absolute bottom-[10%] right-[50%] w-2 h-2 bg-white/16 rounded-full animate-float-delayed"></div>
          <div className="absolute top-[70%] left-[40%] w-1.5 h-1.5 bg-white/24 rounded-full animate-float-slow"></div>
        </div>

        <div className="absolute inset-0 bg-[radial-gradient(circle_at_20%_80%,rgba(255,255,255,0.1),transparent_70%)]"></div>
        <div className="absolute inset-0 bg-[radial-gradient(circle_at_80%_20%,rgba(255,255,255,0.08),transparent_70%)]"></div>
        <div className="absolute inset-0 bg-[radial-gradient(circle_at_40%_40%,rgba(255,255,255,0.05),transparent_70%)]"></div>

        <div className="container mx-auto px-4 relative z-10">
          <div className="grid lg:grid-cols-12 gap-16 items-center">
            <div className="lg:col-span-7 animate-fade-in">
              <div className="mb-8">
                <div className="inline-flex items-center gap-3 bg-black/30 backdrop-blur-2xl border border-white/20 rounded-full px-6 py-3 mb-8 shadow-2xl">
                  <div className="w-2 h-2 bg-green-400 rounded-full animate-pulse shadow-lg shadow-green-400/50"></div>
                  <span className="text-sm font-medium text-white">Available for new projects</span>
                </div>
              </div>

              <h1 className="text-5xl md:text-7xl lg:text-8xl font-bold leading-[0.9] text-balance mb-8">
                <span className="block text-white">I create</span>
                <span className="block text-white">digital</span>
                <span className="block text-white">experiences</span>
              </h1>

              <p className="text-xl md:text-2xl text-white/70 mb-12 text-pretty leading-relaxed max-w-2xl">
                Transforming businesses through innovative web design, cutting-edge AI integration, and strategic
                digital solutions that drive measurable results.
              </p>

              <div className="flex flex-col sm:flex-row gap-6">
                <Link href="/#projects">
                  <Button
                    size="lg"
                    className="group relative bg-gradient-to-r from-white/15 to-white/10 backdrop-blur-xl border border-white/30 hover:border-white/50 text-white rounded-xl px-12 py-4 text-lg font-semibold transition-all duration-500 hover:scale-105 hover:shadow-2xl hover:shadow-white/20 overflow-hidden"
                  >
                    <span className="relative z-10 flex items-center">
                      View My Work
                      <ArrowRight className="ml-3 h-5 w-5 group-hover:translate-x-2 transition-transform duration-500" />
                    </span>
                    <div className="absolute inset-0 bg-gradient-to-r from-white/25 to-white/15 opacity-0 group-hover:opacity-100 transition-opacity duration-500"></div>
                  </Button>
                </Link>
              </div>
            </div>

            <div className="lg:col-span-5 animate-scale-in">
              <div className="relative">
                <Card className="relative p-8 bg-black/20 backdrop-blur-2xl border border-white/20 rounded-[3rem] group hover:translate-y-[-4px] transition-all duration-500">
                  <div className="flex items-center gap-4 mb-6">
                    <div className="flex gap-3">
                      <div className="w-4 h-4 rounded-full bg-red-500 shadow-lg shadow-red-500/50 animate-pulse"></div>
                      <div
                        className="w-4 h-4 rounded-full bg-yellow-500 shadow-lg shadow-yellow-500/50 animate-pulse"
                        style={{ animationDelay: "0.5s" }}
                      ></div>
                      <div
                        className="w-4 h-4 rounded-full bg-green-500 shadow-lg shadow-green-500/50 animate-pulse"
                        style={{ animationDelay: "1s" }}
                      ></div>
                    </div>
                    <div className="text-sm text-white bg-black/50 px-4 py-2 rounded-full backdrop-blur-sm border border-white/20">
                      kristijan.digital/portfolio
                    </div>
                  </div>

                  <div className="font-mono text-sm space-y-3 bg-black/30 p-6 rounded-3xl backdrop-blur-sm border border-white/10 group-hover:border-white/20 transition-all duration-500 min-h-[200px]">
                    {codeLines.slice(0, codeIndex + 1).map((line, index) => (
                      <div
                        key={index}
                        className={`transition-all duration-500 ${index === codeIndex ? "animate-pulse" : ""}`}
                      >
                        {index === 0 && (
                          <div className="flex items-center gap-2">
                            <span className="text-purple-400">const</span>
                            <span className="text-yellow-400">portfolio</span>
                            <span className="text-white">=</span>
                            <span className="text-white">{`{`}</span>
                          </div>
                        )}
                        {index > 0 && index < codeLines.length - 1 && (
                          <div className="pl-4">
                            <span className="text-cyan-400">{line.split(":")[0].trim()}</span>
                            <span className="text-white">:</span>
                            <span className="text-green-400">{line.split(":")[1]}</span>
                          </div>
                        )}
                        {index === codeLines.length - 1 && (
                          <div>
                            <span className="text-white">{`}`}</span>
                          </div>
                        )}
                      </div>
                    ))}
                    <div className="text-white/70 mt-4">// Transforming ideas into reality</div>
                  </div>
                </Card>
              </div>
            </div>
          </div>
        </div>
      </section>

      <section id="about" className="py-32 relative bg-white">
        <div className="container mx-auto px-4 relative z-10">
          <div className="grid lg:grid-cols-12 gap-20 items-center">
            <div className="lg:col-span-5 animate-slide-up">
              <div className="relative group">
                <div className="absolute -inset-4 bg-gradient-to-br from-gray-500/20 via-gray-400/10 to-gray-500/20 rounded-[3rem] blur-2xl group-hover:blur-3xl transition-all duration-700"></div>
                <div className="relative">
                  <Image
                    src="https://i.ibb.co/GXCnB7Y/aboutimg.png"
                    alt="Kristijan Design"
                    width={600}
                    height={700}
                    className="rounded-[3rem] shadow-2xl transition-transform duration-700 group-hover:scale-105 relative z-10 border border-gray-200"
                  />
                  <div className="absolute inset-0 rounded-[3rem] bg-gradient-to-br from-white/5 to-transparent"></div>
                </div>

                {/* Floating Stats Card */}
                <div className="absolute -bottom-8 -right-8 bg-white/90 backdrop-blur-3xl p-8 rounded-3xl shadow-2xl border-2 border-gray-200 hover:shadow-gray-500/30 transition-all duration-700 hover:scale-110 z-20">
                  <div className="grid grid-cols-3 gap-6 text-center">
                    <div className="group/stat">
                      <div className="text-3xl font-bold text-gray-900 group-hover/stat:scale-125 transition-transform duration-500">
                        50+
                      </div>
                      <div className="text-sm text-gray-600 font-medium">Projects</div>
                    </div>
                    <div className="group/stat">
                      <div className="text-3xl font-bold text-gray-900 group-hover/stat:scale-125 transition-transform duration-500">
                        30+
                      </div>
                      <div className="text-sm text-gray-600 font-medium">Clients</div>
                    </div>
                    <div className="group/stat">
                      <div className="text-3xl font-bold text-gray-900 group-hover/stat:scale-125 transition-transform duration-500">
                        5+
                      </div>
                      <div className="text-sm text-gray-600 font-medium">Years</div>
                    </div>
                  </div>
                </div>
              </div>
            </div>

            <div className="lg:col-span-7 animate-slide-up">
              <div className="mb-12">
                <h2 className="text-4xl md:text-6xl font-bold mb-6 leading-tight text-black">
                  Digital Innovation
                  <br />
                  Specialist
                </h2>
                <div className="w-24 h-1 bg-gray-900 rounded-full shadow-lg"></div>
              </div>

              <div className="space-y-8 mb-12">
                <p className="text-xl text-gray-700 leading-relaxed">
                  I'm Kristijan, a passionate digital creator with over 5 years of experience crafting innovative web
                  solutions that transform businesses. I specialize in integrating cutting-edge AI technologies with
                  creative design to deliver exceptional user experiences.
                </p>
                <p className="text-lg text-gray-600 leading-relaxed">
                  My philosophy centers on collaboration and innovation. I believe in building strong partnerships with
                  clients to understand their unique vision and challenges, creating digital solutions that not only
                  look stunning but leverage the latest technologies to give your business a competitive edge.
                </p>
              </div>

              <div className="grid grid-cols-2 gap-6 mb-12">
                {[
                  { icon: Code, title: "Web Development", desc: "Modern, scalable solutions" },
                  { icon: Palette, title: "UI/UX Design", desc: "User-centered experiences" },
                  { icon: Brain, title: "AI Integration", desc: "Intelligent automation" },
                  { icon: Rocket, title: "Digital Strategy", desc: "Growth-focused approach" },
                ].map((skill, index) => (
                  <div
                    key={index}
                    className="flex items-start gap-4 bg-gray-50 p-6 rounded-3xl hover:bg-gray-100 hover:border-gray-300 border border-gray-200 transition-all duration-500 group/skill hover:scale-105"
                  >
                    <div className="bg-gray-100 p-3 rounded-2xl shadow-lg group-hover/skill:shadow-gray-300 transition-all duration-500 group-hover/skill:scale-125 border border-gray-200">
                      {React.createElement(skill.icon, { className: "h-6 w-6 text-gray-700" })}
                    </div>
                    <div>
                      <h3 className="font-semibold text-black mb-1 group-hover/skill:text-gray-700 transition-colors duration-500">
                        {skill.title}
                      </h3>
                      <p className="text-sm text-gray-600">{skill.desc}</p>
                    </div>
                  </div>
                ))}
              </div>

              <Link href="/contact">
                <Button
                  size="lg"
                  className="bg-black hover:bg-gray-800 shadow-2xl hover:shadow-gray-500/30 transition-all duration-700 hover:scale-110 rounded-2xl px-10 py-4 text-lg font-semibold text-white"
                >
                  Let's Collaborate
                  <ArrowRight className="ml-3 h-5 w-5" />
                </Button>
              </Link>
            </div>
          </div>
        </div>
      </section>

      <section id="expertise" className="py-32 relative bg-white">
        <div className="container mx-auto px-4 relative z-10">
          <div className="text-center mb-20">
            <h2 className="text-4xl md:text-6xl font-bold mb-6 text-black">My Expertise</h2>
            <div className="w-24 h-1 bg-black mx-auto rounded-full shadow-lg mb-6"></div>
            <p className="text-xl text-gray-600 max-w-3xl mx-auto leading-relaxed">
              Combining creative design with cutting-edge technology to deliver exceptional digital experiences
            </p>
          </div>

          <div className="grid md:grid-cols-2 lg:grid-cols-3 gap-8">
            {[
              {
                slug: "web-design",
                icon: Palette,
                title: "Web Design & UI/UX",
                description:
                  "Creating stunning, user-centric websites that captivate audiences and drive conversions. Every design is meticulously crafted to reflect your brand identity while ensuring optimal user experience across all devices.",
              },
              {
                slug: "web-development",
                icon: Code,
                title: "Web Development",
                description:
                  "Building robust, scalable web applications using cutting-edge technologies and best practices. From simple websites to complex platforms, ensuring fast, secure, and maintainable digital solutions.",
              },
              {
                slug: "ai-integration",
                icon: Brain,
                title: "AI Integration",
                description:
                  "Harnessing artificial intelligence to create smart, adaptive digital solutions. Implementing AI-powered features like chatbots, predictive analytics, and automation that add real business value.",
              },
              {
                slug: "ecommerce",
                icon: Zap,
                title: "E-commerce Solutions",
                description:
                  "Developing powerful online stores with advanced features, secure payment systems, and optimized user journeys that maximize conversions and provide seamless shopping experiences.",
              },
              {
                slug: "digital-strategy",
                icon: Target,
                title: "Digital Strategy",
                description:
                  "Providing strategic guidance to help businesses navigate the digital landscape effectively. Data-driven approach ensures digital investments deliver measurable results and sustainable growth.",
              },
              {
                slug: "performance-optimization",
                icon: Rocket,
                title: "Performance Optimization",
                description:
                  "Optimizing websites and applications for maximum speed, efficiency, and user satisfaction. Advanced techniques and monitoring ensure your digital presence performs at its peak potential.",
              },
            ].map((service, index) => (
              <Link key={index} href={`/services/${service.slug}`}>
                <Card className="p-8 bg-white border-2 border-gray-200 shadow-xl hover:shadow-2xl hover:shadow-gray-300/50 transition-all duration-700 hover:scale-105 hover:-translate-y-4 group rounded-3xl relative overflow-hidden cursor-pointer">
                  <div className="relative z-10">
                    <div className="bg-gray-100 p-4 rounded-3xl shadow-lg group-hover:shadow-gray-300/50 transition-all duration-700 group-hover:scale-125 group-hover:rotate-12 border border-gray-200 mb-6 w-fit">
                      {React.createElement(service.icon, { className: "h-8 w-8 text-gray-700" })}
                    </div>

                    <h3 className="text-2xl font-bold mb-4 group-hover:text-gray-600 transition-colors duration-500 text-black">
                      {service.title}
                    </h3>

                    <p className="text-gray-600 leading-relaxed mb-4">{service.description}</p>

                    <div className="flex items-center text-black font-semibold group-hover:text-gray-600 transition-colors duration-500">
                      Learn More
                      <ArrowRight className="ml-2 h-4 w-4 group-hover:translate-x-2 transition-transform duration-500" />
                    </div>
                  </div>

                  <div className="absolute inset-0 bg-gradient-to-br from-gray-50/50 via-gray-100/30 to-gray-50/50 opacity-0 group-hover:opacity-100 transition-all duration-700 rounded-3xl"></div>
                  <div className="absolute inset-0 border-2 border-transparent group-hover:border-gray-300/50 rounded-3xl transition-all duration-700"></div>
                </Card>
              </Link>
            ))}
          </div>
        </div>
      </section>

      <section className="py-32 relative overflow-hidden bg-[#161d25]">
        <div className="absolute inset-0 overflow-hidden">
          <div className="absolute top-[20%] left-[25%] w-2 h-2 bg-white/20 rounded-full animate-float"></div>
          <div className="absolute top-[40%] right-[35%] w-1 h-1 bg-white/30 rounded-full animate-float-delayed"></div>
          <div className="absolute bottom-[30%] left-[40%] w-3 h-3 bg-white/15 rounded-full animate-float-slow"></div>
          <div className="absolute top-[70%] right-[20%] w-1.5 h-1.5 bg-white/25 rounded-full animate-float"></div>
          <div className="absolute bottom-[50%] right-[70%] w-2 h-2 bg-white/18 rounded-full animate-float-delayed"></div>
          <div className="absolute top-[60%] left-[10%] w-1 h-1 bg-white/22 rounded-full animate-float-slow"></div>
          <div className="absolute top-[15%] left-[75%] w-2.5 h-2.5 bg-white/16 rounded-full animate-float"></div>
          <div className="absolute bottom-[15%] right-[60%] w-1 h-1 bg-white/28 rounded-full animate-float-delayed"></div>
          <div className="absolute top-[85%] left-[55%] w-1.5 h-1.5 bg-white/24 rounded-full animate-float-slow"></div>
          <div className="absolute bottom-[70%] right-[45%] w-1 h-1 bg-white/26 rounded-full animate-float"></div>
        </div>

        <div className="absolute inset-0 bg-[radial-gradient(circle_at_30%_70%,rgba(255,255,255,0.08),transparent_70%)]"></div>
        <div className="absolute inset-0 bg-[radial-gradient(circle_at_70%_30%,rgba(255,255,255,0.06),transparent_70%)]"></div>

        <div className="container mx-auto px-4 relative z-10">
          <div className="text-center mb-20">
            <div className="inline-flex items-center gap-2 bg-white/5 backdrop-blur-2xl px-4 py-2 rounded-full mb-6 border border-white/20">
              <div className="w-2 h-2 bg-white rounded-full animate-pulse"></div>
              <span className="text-sm font-medium text-white/90">Why Choose Me</span>
            </div>
            <h2 className="text-4xl md:text-6xl font-bold mb-6 text-white leading-tight">Why Work With Me?</h2>
            <div className="w-24 h-1 bg-white mx-auto rounded-full shadow-lg mb-6"></div>
            <p className="text-xl text-white/70 max-w-3xl mx-auto leading-relaxed">
              Discover what sets me apart and why clients choose me for their digital transformation journey
            </p>
          </div>

          <div className="grid md:grid-cols-2 lg:grid-cols-4 gap-8">
            {[
              {
                icon: Award,
                title: "Proven Excellence",
                description:
                  "5+ years of experience with 50+ successful projects and a 98% client satisfaction rate. My work consistently delivers measurable results and exceeds expectations.",
                metric: "98%",
                metricLabel: "Client Satisfaction",
              },
              {
                icon: TrendingUp,
                title: "Measurable Results",
                description:
                  "I focus on ROI-driven solutions that deliver real business impact. My clients see average increases of 250% in conversions and 180% in user engagement.",
                metric: "250%",
                metricLabel: "Avg. Conversion Increase",
              },
              {
                icon: Clock,
                title: "On-Time Delivery",
                description:
                  "I respect your time and deadlines. With a proven track record of delivering projects on schedule, you can trust me to meet your launch dates and milestones.",
                metric: "100%",
                metricLabel: "On-Time Delivery",
              },
              {
                icon: Users,
                title: "Dedicated Partnership",
                description:
                  "I believe in building lasting relationships. You get direct access to me throughout the project, with regular updates and transparent communication every step of the way.",
                metric: "24/7",
                metricLabel: "Support Available",
              },
            ].map((reason, index) => (
              <Card
                key={index}
                className="p-8 bg-white/5 backdrop-blur-2xl border border-white/20 shadow-2xl hover:shadow-white/10 transition-all duration-700 hover:scale-105 hover:-translate-y-4 group rounded-3xl relative overflow-hidden"
              >
                <div className="absolute inset-0 bg-gradient-to-br from-white/10 via-white/5 to-transparent opacity-0 group-hover:opacity-100 transition-all duration-700 rounded-3xl"></div>

                <div className="relative z-10">
                  <div className="bg-white/10 backdrop-blur-sm p-4 rounded-3xl shadow-lg group-hover:shadow-white/20 transition-all duration-700 group-hover:scale-125 group-hover:rotate-12 border border-white/20 mb-6 w-fit">
                    {React.createElement(reason.icon, { className: "h-8 w-8 text-white" })}
                  </div>

                  <div className="mb-4">
                    <div className="text-3xl font-bold text-white mb-1">{reason.metric}</div>
                    <div className="text-sm text-white/60 font-medium">{reason.metricLabel}</div>
                  </div>

                  <h3 className="text-xl font-bold mb-4 group-hover:text-white/90 transition-colors duration-500 text-white">
                    {reason.title}
                  </h3>

                  <p className="text-white/70 leading-relaxed">{reason.description}</p>
                </div>
              </Card>
            ))}
          </div>
        </div>
      </section>

      <section className="py-32 relative bg-white">
        <div className="container mx-auto px-4 relative z-10">
          <div className="text-center mb-20">
            <h2 className="text-4xl md:text-6xl font-bold mb-6 text-black">My Process</h2>
            <div className="w-24 h-1 bg-black mx-auto rounded-full shadow-lg mb-6"></div>
            <p className="text-xl text-gray-600 max-w-3xl mx-auto leading-relaxed">
              A proven methodology that ensures exceptional results and seamless collaboration from concept to launch
            </p>
          </div>

          <div className="grid lg:grid-cols-4 gap-8 relative">
            {/* Connection Lines */}
            <div className="hidden lg:block absolute top-24 left-0 right-0 h-0.5 bg-gradient-to-r from-gray-200 via-gray-300 to-gray-200"></div>

            {[
              {
                step: "01",
                title: "Discovery & Planning",
                description:
                  "We start by understanding your vision, goals, and requirements to create a strategic roadmap for your project.",
              },
              {
                step: "02",
                title: "Design & Prototyping",
                description:
                  "Creating stunning visual designs and interactive prototypes that bring your ideas to life with user-centered approach.",
              },
              {
                step: "03",
                title: "Development & Integration",
                description:
                  "Building your solution using cutting-edge technologies, implementing features and ensuring scalable architecture.",
              },
              {
                step: "04",
                title: "Launch & Optimization",
                description:
                  "Deploying your project live with thorough testing, optimization, and ongoing support to ensure continued success.",
              },
            ].map((phase, index) => (
              <div key={index} className="relative group">
                {/* Step Number Circle */}
                <div className="relative z-10 mx-auto w-16 h-16 bg-white border-4 border-gray-200 rounded-full flex items-center justify-center mb-6 group-hover:border-gray-400 group-hover:scale-110 transition-all duration-500 shadow-lg">
                  <span className="text-lg font-bold text-gray-700 group-hover:text-black transition-colors duration-300">
                    {phase.step}
                  </span>
                </div>

                {/* Content Card */}
                <Card className="p-6 bg-white border-2 border-gray-200 shadow-lg hover:shadow-2xl hover:shadow-gray-300/30 transition-all duration-500 hover:scale-105 hover:-translate-y-2 group/card rounded-2xl relative overflow-hidden">
                  <div className="absolute inset-0 bg-gradient-to-br from-gray-50/50 via-gray-100/30 to-gray-50/50 opacity-0 group-hover/card:opacity-100 transition-all duration-500 rounded-2xl"></div>

                  <div className="relative z-10">
                    <h3 className="text-xl font-bold text-black mb-4 group-hover/card:text-gray-700 transition-colors duration-300">
                      {phase.title}
                    </h3>

                    <p className="text-gray-600 leading-relaxed text-sm">{phase.description}</p>
                  </div>
                </Card>
              </div>
            ))}
          </div>

          <div className="text-center mt-16">
            <Link href="/contact">
              <Button
                size="lg"
                className="bg-black hover:bg-gray-800 shadow-2xl hover:shadow-gray-500/30 transition-all duration-700 hover:scale-110 rounded-2xl px-12 py-4 text-lg font-semibold text-white"
              >
                Start Your Project
                <ArrowRight className="ml-3 h-5 w-5" />
              </Button>
            </Link>
          </div>
        </div>
      </section>

      <section id="projects" className="py-32 relative bg-gray-50">
        <div className="container mx-auto px-4 relative z-10">
          <div className="text-center mb-20">
            <h2 className="text-4xl md:text-6xl font-bold mb-6 text-black">Featured Projects</h2>
            <div className="w-24 h-1 bg-black mx-auto rounded-full shadow-lg mb-6"></div>
            <p className="text-xl text-gray-600 max-w-3xl mx-auto leading-relaxed">
              Explore my latest work showcasing innovative solutions and creative excellence
            </p>
          </div>

          <div className="flex flex-wrap justify-center gap-4 mb-16">
            {[
              { key: "all", label: "All Works" },
              { key: "websites", label: "Websites" },
              { key: "applications", label: "Applications" },
              { key: "social-media", label: "Social Media Design" },
              { key: "ai-projects", label: "AI Projects" },
            ].map((filter) => (
              <Button
                key={filter.key}
                variant={activeFilter === filter.key ? "default" : "outline"}
                onClick={() => setActiveFilter(filter.key)}
                className={`rounded-full px-8 py-3 font-semibold transition-all duration-500 hover:scale-105 ${
                  activeFilter === filter.key
                    ? "bg-black text-white shadow-2xl hover:bg-gray-800"
                    : "bg-white text-black border-2 border-gray-200 hover:border-gray-400 hover:bg-gray-50"
                }`}
              >
                {filter.label}
              </Button>
            ))}
          </div>

          <div className="grid md:grid-cols-2 lg:grid-cols-3 gap-8">
            {filteredProjects.slice(0, 6).map((project, index) => (
              <Link key={project.id} href={`/projects/${project.id}`}>
                <Card className="group overflow-hidden bg-white border-2 border-gray-200 shadow-xl hover:shadow-2xl hover:shadow-gray-300/50 transition-all duration-700 hover:scale-105 hover:-translate-y-4 rounded-3xl cursor-pointer h-full flex flex-col">
                  <div className="relative overflow-hidden">
                    <Image
                      src={project.image || "/placeholder.svg"}
                      alt={project.title}
                      width={600}
                      height={400}
                      className="w-full h-64 object-cover transition-transform duration-700 group-hover:scale-110"
                    />
                    <div className="absolute inset-0 bg-black/0 group-hover:bg-black/20 transition-all duration-500"></div>
                    <div className="absolute inset-0 flex items-center justify-center opacity-0 group-hover:opacity-100 transition-all duration-500">
                      <div className="bg-white/90 backdrop-blur-sm px-6 py-3 rounded-full shadow-lg">
                        <span className="text-black font-semibold">View Project</span>
                      </div>
                    </div>
                  </div>

                  <div className="p-8 flex-1 flex flex-col">
                    <div className="flex items-center gap-2 mb-4">
                      <span className="text-xs font-semibold text-gray-500 bg-gray-100 px-3 py-1 rounded-full">
                        {project.year}
                      </span>
                      <span className="text-xs font-semibold text-gray-500 bg-gray-100 px-3 py-1 rounded-full">
                        {project.client}
                      </span>
                    </div>

                    <h3 className="text-xl font-bold text-black mb-3 group-hover:text-gray-700 transition-colors duration-500">
                      {project.title}
                    </h3>

                    <p className="text-gray-600 leading-relaxed mb-4 text-sm flex-1">{project.description}</p>

                    <div className="flex flex-wrap gap-2 mt-auto">
                      {project.technologies.slice(0, 3).map((tech, techIndex) => (
                        <span
                          key={techIndex}
                          className="text-xs font-medium text-gray-600 bg-gray-100 px-3 py-1 rounded-full"
                        >
                          {tech}
                        </span>
                      ))}
                      {project.technologies.length > 3 && (
                        <span className="text-xs font-medium text-gray-500 bg-gray-50 px-3 py-1 rounded-full">
                          +{project.technologies.length - 3} more
                        </span>
                      )}
                    </div>
                  </div>
                </Card>
              </Link>
            ))}
          </div>

          <div className="text-center mt-16">
            <Link href="/projects">
              <Button
                size="lg"
                className="bg-black hover:bg-gray-800 shadow-2xl hover:shadow-gray-500/30 transition-all duration-700 hover:scale-110 rounded-2xl px-12 py-4 text-lg font-semibold text-white"
              >
                View All Projects
                <ArrowRight className="ml-3 h-5 w-5" />
              </Button>
            </Link>
          </div>
        </div>
      </section>

      <section id="testimonials" className="py-32 relative bg-gradient-to-br from-gray-50 to-white overflow-hidden">
        <div className="absolute inset-0">
          <div className="absolute top-[20%] left-[15%] w-72 h-72 bg-blue-100/30 rounded-full blur-3xl"></div>
          <div className="absolute bottom-[20%] right-[15%] w-96 h-96 bg-purple-100/20 rounded-full blur-3xl"></div>
        </div>
        
        <div className="container mx-auto px-4 relative z-10">
          <div className="text-center mb-20">
            <h2 className="text-4xl md:text-6xl font-bold mb-6 text-black">Client Testimonials</h2>
            <div className="w-24 h-1 bg-black mx-auto rounded-full shadow-lg mb-6"></div>
            <p className="text-xl text-gray-600 max-w-3xl mx-auto leading-relaxed">
              Hear what my clients say about working with me and the results we've achieved together
            </p>
          </div>

          <div className="relative w-full overflow-hidden">
            <div className="py-16">
              <div
                className="flex transition-transform duration-700 ease-out"
                style={{ transform: `translateX(-${currentTestimonial * 100}%)` }}
              >
                {testimonialGroups.map((group, groupIndex) => (
                  <div key={groupIndex} className="w-full flex-shrink-0">
                    <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-8 px-4">
                    {group.map((testimonial, index) => (
                      <div key={index} className="group relative">
                        {/* Background decoration */}
                        <div className="absolute -inset-4 bg-gradient-to-r from-gray-500/10 via-gray-400/10 to-gray-500/10 rounded-3xl opacity-0 group-hover:opacity-100 transition-all duration-500 blur-xl"></div>
                        
                        <Card className="relative bg-white/90 backdrop-blur-sm border border-gray-200/50 shadow-xl hover:shadow-2xl transition-all duration-500 hover:scale-105 hover:-translate-y-2 rounded-3xl p-8 h-full group-hover:border-gray-300">
                          {/* Quote icon */}
                          <div className="absolute -top-4 -left-4 w-12 h-12 bg-gradient-to-br from-gray-700 to-gray-900 rounded-full flex items-center justify-center shadow-lg">
                            <svg className="w-6 h-6 text-white" fill="currentColor" viewBox="0 0 24 24">
                              <path d="M14.017 21v-7.391c0-5.704 3.731-9.57 8.983-10.609l.995 2.151c-2.432.917-3.995 3.638-3.995 5.849h4v10h-9.983zm-14.017 0v-7.391c0-5.704 3.748-9.57 9-10.609l.996 2.151c-2.433.917-3.996 3.638-3.996 5.849h4v10h-10z"/>
                            </svg>
                          </div>
                          
                          <div className="mt-4">
                            <div className="flex items-center gap-4 mb-6">
                              <div className="relative">
                                <Image
                                  src={testimonial.avatar || "/placeholder.svg"}
                                  alt={testimonial.name}
                                  width={70}
                                  height={70}
                                  className="rounded-full object-cover border-3 border-white shadow-lg"
                                />
                                <div className="absolute -bottom-1 -right-1 w-5 h-5 bg-green-500 border-2 border-white rounded-full"></div>
                              </div>
                              <div className="flex-1">
                                <h4 className="font-bold text-gray-900 text-lg">{testimonial.name}</h4>
                                <p className="text-sm text-gray-600 font-medium">{testimonial.position}</p>
                                <p className="text-xs text-gray-500 bg-gray-100 px-3 py-1 rounded-full inline-block mt-1">{testimonial.company}</p>
                              </div>
                            </div>

                            {/* Star rating */}
                            <div className="flex gap-1 mb-6">
                              {[...Array(testimonial.rating)].map((_, i) => (
                                <svg key={i} className="w-5 h-5 text-yellow-400" fill="currentColor" viewBox="0 0 20 20">
                                  <path d="M9.049 2.927c.3-.921 1.603-.921 1.902 0l1.07 3.292a1 1 0 00.95.69h3.462c.969 0 1.371 1.24.588 1.81l-2.8 2.034a1 1 0 00-.364 1.118l1.07 3.292c.3.921-.755 1.688-1.54 1.118l-2.8-2.034a1 1 0 00-1.175 0l-2.8 2.034c-.784.57-1.838-.197-1.539-1.118l1.07-3.292a1 1 0 00-.364-1.118L2.98 8.72c-.783-.57-.38-1.81.588-1.81h3.461a1 1 0 00.951-.69l1.07-3.292z" />
                                </svg>
                              ))}
                            </div>

                            <blockquote className="text-gray-700 leading-relaxed text-base italic font-medium relative">
                              "{testimonial.content}"
                            </blockquote>
                          </div>
                        </Card>
                      </div>
                    ))}
                    </div>
                  </div>
                ))}
              </div>
            </div>

            {/* Pagination dots */}
            <div className="flex justify-center gap-3 mt-8">
              {testimonialGroups.map((_, index) => (
                <button
                  key={index}
                  onClick={() => setCurrentTestimonial(index)}
                  className={`transition-all duration-300 rounded-full ${
                    currentTestimonial === index 
                      ? "w-8 h-3 bg-gradient-to-r from-gray-700 to-gray-900 shadow-lg" 
                      : "w-3 h-3 bg-gray-300 hover:bg-gray-400"
                  }`}
                />
              ))}
            </div>
          </div>
        </div>
      </section>

      <section className="py-32 relative bg-gray-50">
        <div className="container mx-auto px-4 relative z-10">
          <div className="text-center mb-20">
            <h2 className="text-4xl md:text-6xl font-bold mb-6 text-black">Frequently Asked Questions</h2>
            <div className="w-24 h-1 bg-black mx-auto rounded-full shadow-lg mb-6"></div>
            <p className="text-xl text-gray-600 max-w-3xl mx-auto leading-relaxed">
              Get answers to common questions about my services, process, and approach
            </p>
          </div>

          <div className="max-w-4xl mx-auto space-y-6">
            {faqs.map((faq, index) => (
              <FAQItem
                key={index}
                faq={faq}
                index={index}
                isOpen={openFaq === index}
                onToggle={() => setOpenFaq(openFaq === index ? null : index)}
              />
            ))}
          </div>
        </div>
      </section>

      {/* Ready to Start Section + Footer Combined */}
      <section className="relative py-20 overflow-hidden bg-[#161d25]">
        <div className="absolute inset-0 overflow-hidden">
          <div className="absolute top-[25%] left-[30%] w-2 h-2 bg-white/20 rounded-full animate-float"></div>
          <div className="absolute top-[45%] right-[40%] w-1 h-1 bg-white/30 rounded-full animate-float-delayed"></div>
          <div className="absolute bottom-[35%] left-[45%] w-3 h-3 bg-white/15 rounded-full animate-float-slow"></div>
          <div className="absolute top-[75%] right-[25%] w-1.5 h-1.5 bg-white/25 rounded-full animate-float"></div>
          <div className="absolute bottom-[55%] right-[75%] w-2 h-2 bg-white/18 rounded-full animate-float-delayed"></div>
          <div className="absolute top-[65%] left-[15%] w-1 h-1 bg-white/22 rounded-full animate-float-slow"></div>
        </div>

        <div className="container mx-auto px-4 relative z-10">
          {/* Ready Badge */}
          <div className="text-center mb-12">
            <div className="inline-flex items-center gap-2 bg-white/10 backdrop-blur-xl border border-white/20 rounded-full px-4 py-2 mb-8">
              <div className="w-2 h-2 bg-green-400 rounded-full animate-pulse"></div>
              <span className="text-white/90 text-sm font-medium">Ready to start your project?</span>
            </div>
          </div>

          {/* Main Content */}
          <div className="text-center mb-16">
            <h2 className="text-4xl md:text-6xl font-bold text-white mb-6 leading-tight text-balance">
              Let's Create Something Extraordinary
            </h2>
            <p className="text-xl text-white/70 max-w-3xl mx-auto leading-relaxed mb-12 text-pretty">
              Transform your digital vision into reality with innovative solutions that drive growth, engage users, and
              deliver exceptional results for your business.
            </p>
            <Link href="/contact">
              <Button
                size="lg"
                className="bg-white text-black hover:bg-white/90 rounded-full px-8 py-4 text-lg font-semibold transition-all duration-300 hover:scale-105 hover:shadow-xl"
              >
                Start Your Project
                <ArrowRight className="ml-3 h-5 w-5" />
              </Button>
            </Link>
          </div>

          {/* Social Icons */}
          <div className="flex items-center justify-center gap-6 mb-16">
            <a
              href="https://linkedin.com/in/kristijan-digital"
              target="_blank"
              rel="noopener noreferrer"
              className="w-12 h-12 bg-white/10 backdrop-blur-xl border border-white/20 rounded-full flex items-center justify-center text-white/70 hover:text-white hover:bg-white/20 transition-all duration-300 hover:scale-110"
            >
              <svg className="w-5 h-5" fill="currentColor" viewBox="0 0 24 24">
                <path d="M20.447 20.452h-3.554v-5.569c0-1.328-.027-3.037-1.852-3.037-1.853 0-2.136 1.445-2.136 2.939v5.667H9.351V9h3.414v1.561h.046c.477-.9 1.637-1.85 3.37-1.85 3.601 0 4.267 2.37 4.267 5.455v6.286zM5.337 7.433c-1.144 0-2.063-.926-2.063-2.065 0-1.138.92-2.063 2.063-2.063 1.14 0 2.064.925 2.064 2.063 0 1.139-.925 2.065-2.064 2.065zm1.782 13.019H3.555V9h3.564v11.452zM22.225 0H1.771C.792 0 0 .774 0 1.729v20.542C0 23.227.792 24 1.771 24h20.451C23.2 24 24 23.227 24 22.271V1.729C24 .774 23.2 0 22.222 0h.003z" />
              </svg>
            </a>
            <a
              href="https://github.com/kristijan-digital"
              target="_blank"
              rel="noopener noreferrer"
              className="w-12 h-12 bg-white/10 backdrop-blur-xl border border-white/20 rounded-full flex items-center justify-center text-white/70 hover:text-white hover:bg-white/20 transition-all duration-300 hover:scale-110"
            >
              <svg className="w-5 h-5" fill="currentColor" viewBox="0 0 24 24">
                <path d="M12 0c-6.626 0-12 5.373-12 12 0 5.302 3.438 9.8 8.207 11.387.599.111.793-.261.793-.577v-2.234c-3.338.726-4.033-1.416-4.033-1.416-.546-1.387-1.333-1.756-1.333-1.756-1.089-.745.083-.729.083-.729 1.205.084 1.839 1.237 1.839 1.237 1.07 1.834 2.807 1.304 3.492.997.107-.775.418-1.305.762-1.604-2.665-.305-5.467-1.334-5.467-5.931 0-1.311.469-2.381 1.236-3.221-.124-.303-.535-1.524.117-3.176 0 0 1.008-.322 3.301 1.23.957-.266 1.983-.399 3.003-.404 1.02.005 2.047.138 3.006.404 2.291-1.552 3.297-1.23 3.297-1.23.653 1.653.242 2.874.118 3.176.77.84 1.235 1.911 1.235 3.221 0 4.609-2.807 5.624-5.479 5.921.43.372.823 1.102.823 2.222v3.293c0 .319.192.694.801.576 4.765-1.589 8.199-6.086 8.199-11.386 0-6.627-5.373-12-12-12z" />
              </svg>
            </a>
            <a
              href="https://instagram.com/kristijan.digital"
              target="_blank"
              rel="noopener noreferrer"
              className="w-12 h-12 bg-white/10 backdrop-blur-xl border border-white/20 rounded-full flex items-center justify-center text-white/70 hover:text-white hover:bg-white/20 transition-all duration-300 hover:scale-110"
            >
              <svg className="w-5 h-5" fill="currentColor" viewBox="0 0 24 24">
                <path d="M12 2.163c3.204 0 3.584.012 4.85.07 3.252.148 4.771 1.691 4.919 4.919.058 1.265.069 1.645.069 4.849 0 3.205-.012 3.584-.069 4.849-.149 3.225-1.664 4.771-4.919 4.919-1.266.058-1.644.07-4.85.07-3.204 0-3.584-.012-4.849-.07-3.26-.149-4.771-1.699-4.919-4.92-.058-1.265-.07-1.644-.07-4.849 0-3.204.013-3.583.07-4.849.149-3.227 1.664-4.771 4.919-4.919 1.266-.057 1.645-.069 4.849-.069zm0-2.163c-3.259 0-3.667.014-4.947.072-4.358.2-6.78 2.618-6.98 6.98-.059 1.281-.073 1.689-.073 4.948 0 3.259.014 3.668.072 4.948.2 4.358 2.618 6.78 6.98 6.98 1.281.058 1.689.072 4.948.072 3.259 0 3.668-.014 4.948-.072 4.354-.2 6.782-2.618 6.979-6.98.059-1.28.073-1.689.073-4.948 0-3.259-.014-3.667-.072-4.947-.196-4.354-2.617-6.78-6.979-6.98-1.281-.059-1.69-.073-4.949-.073zm0 5.838c-3.403 0-6.162 2.759-6.162 6.162s2.759 6.163 6.162 6.163 6.162-2.759 6.162-6.163c0-3.403-2.759-6.162-6.162-6.162zm0 10.162c-2.209 0-4-1.79-4-4 0-2.209 1.791-4 4-4s4 1.791 4 4c0 2.21-1.791 4-4 4zm6.406-11.845c-.796 0-1.441.645-1.441 1.44s.645 1.44 1.441 1.44c.795 0 1.439-.645 1.439-1.44s-.644-1.44-1.439-1.44z" />
              </svg>
            </a>
            <a
              href="mailto:hello@kristijan.digital"
              className="w-12 h-12 bg-white/10 backdrop-blur-xl border border-white/20 rounded-full flex items-center justify-center text-white/70 hover:text-white hover:bg-white/20 transition-all duration-300 hover:scale-110"
            >
              <svg className="w-5 h-5" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                <path
                  strokeLinecap="round"
                  strokeLinejoin="round"
                  strokeWidth={2}
                  d="M3 8l7.89 4.26a2 2 0 002.22 0L21 8M5 19h14a2 2 0 002-2V7a2 2 0 00-2-2H5a2 2 0 00-2 2v10a2 2 0 002 2z"
                />
              </svg>
            </a>
          </div>

          {/* Copyright */}
          <div className="text-center">
            <p className="text-white/60 text-sm">© 2024 Kristijan Design. All rights reserved.</p>
            <p className="text-white/60 text-sm mt-2 flex items-center justify-center gap-2">
              Crafted with <span className="text-white animate-pulse">❤️</span> and innovation
            </p>
          </div>
        </div>
      </section>
    </div>
  )
}
