"use strict";
var __createBinding = (this && this.__createBinding) || (Object.create ? (function(o, m, k, k2) {
    if (k2 === undefined) k2 = k;
    Object.defineProperty(o, k2, { enumerable: true, get: function() { return m[k]; } });
}) : (function(o, m, k, k2) {
    if (k2 === undefined) k2 = k;
    o[k2] = m[k];
}));
var __setModuleDefault = (this && this.__setModuleDefault) || (Object.create ? (function(o, v) {
    Object.defineProperty(o, "default", { enumerable: true, value: v });
}) : function(o, v) {
    o["default"] = v;
});
var __importStar = (this && this.__importStar) || function (mod) {
    if (mod && mod.__esModule) return mod;
    var result = {};
    if (mod != null) for (var k in mod) if (k !== "default" && Object.prototype.hasOwnProperty.call(mod, k)) __createBinding(result, mod, k);
    __setModuleDefault(result, mod);
    return result;
};
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.runBuiltInOptimize = void 0;
const cheerio_1 = __importDefault(require("cheerio"));
const esbuild = __importStar(require("esbuild"));
const fs_1 = require("fs");
const glob_1 = require("glob");
const mkdirp_1 = __importDefault(require("mkdirp"));
const path_1 = __importDefault(require("path"));
const rimraf_1 = __importDefault(require("rimraf"));
const logger_1 = require("../logger");
const util_1 = require("../util");
const file_urls_1 = require("./file-urls");
// We want to output our bundled build directly into our build directory, but esbuild
// has a bug where it complains about overwriting source files even when write: false.
// We create a fake bundle directory for now. Nothing ever actually gets written here.
const FAKE_BUILD_DIRECTORY = path_1.default.join(util_1.PROJECT_CACHE_DIR, '~~bundle~~');
const FAKE_BUILD_DIRECTORY_REGEX = /.*\~\~bundle\~\~[\\\/]/;
/**
 * Scan a directory and remove any empty folders, recursively.
 */
async function removeEmptyFolders(directoryLoc) {
    if (!(await fs_1.promises.stat(directoryLoc)).isDirectory()) {
        return false;
    }
    // If folder is empty, clear it
    const files = await fs_1.promises.readdir(directoryLoc);
    if (files.length === 0) {
        await fs_1.promises.rmdir(directoryLoc);
        return false;
    }
    // Otherwise, step in and clean each contained item
    await Promise.all(files.map((file) => removeEmptyFolders(path_1.default.join(directoryLoc, file))));
    // After, check again if folder is now empty
    const afterFiles = await fs_1.promises.readdir(directoryLoc);
    if (afterFiles.length == 0) {
        await fs_1.promises.rmdir(directoryLoc);
    }
    return true;
}
/** Collect deep imports in the given set, recursively. */
function collectDeepImports(url, manifest, set) {
    if (set.has(url)) {
        return;
    }
    set.add(url);
    const manifestEntry = manifest.inputs[url];
    if (!manifestEntry) {
        throw new Error('Not Found in manifest: ' + url);
    }
    manifestEntry.imports.forEach(({ path }) => collectDeepImports(path, manifest, set));
    return;
}
/**
 * Scan a collection of HTML files for entrypoints. A file is deemed an "html entrypoint"
 * if it contains an <html> element. This prevents partials from being scanned.
 */
async function scanHtmlEntrypoints(htmlFiles) {
    return Promise.all(htmlFiles.map(async (htmlFile) => {
        const code = await fs_1.promises.readFile(htmlFile, 'utf-8');
        const root = cheerio_1.default.load(code);
        const isHtmlFragment = root.html().startsWith('<html><head></head><body>');
        if (isHtmlFragment) {
            return null;
        }
        return {
            file: htmlFile,
            root,
            getScripts: () => root('script[type="module"]'),
            getStyles: () => root('style'),
            getLinks: (rel) => root(`link[rel="${rel}"]`),
        };
    }));
}
async function extractBaseUrl(htmlData, baseUrl) {
    const { root, getScripts, getLinks } = htmlData;
    if (!baseUrl || baseUrl === '/') {
        return;
    }
    getScripts().each((_, elem) => {
        const scriptRoot = root(elem);
        const scriptSrc = scriptRoot.attr('src');
        if (!scriptSrc || !scriptSrc.startsWith(baseUrl)) {
            return;
        }
        scriptRoot.attr('src', util_1.addLeadingSlash(scriptSrc.replace(baseUrl, '')));
        scriptRoot.attr('snowpack-baseurl', 'true');
    });
    getLinks('stylesheet').each((_, elem) => {
        const linkRoot = root(elem);
        const styleHref = linkRoot.attr('href');
        if (!styleHref || !styleHref.startsWith(baseUrl)) {
            return;
        }
        linkRoot.attr('href', util_1.addLeadingSlash(styleHref.replace(baseUrl, '')));
        linkRoot.attr('snowpack-baseurl', 'true');
    });
}
async function restitchBaseUrl(htmlData, baseUrl) {
    const { root, getScripts, getLinks } = htmlData;
    getScripts()
        .filter('[snowpack-baseurl]')
        .each((_, elem) => {
        const scriptRoot = root(elem);
        const scriptSrc = scriptRoot.attr('src');
        scriptRoot.attr('src', util_1.removeTrailingSlash(baseUrl) + util_1.addLeadingSlash(scriptSrc));
        scriptRoot.removeAttr('snowpack-baseurl');
    });
    getLinks('stylesheet')
        .filter('[snowpack-baseurl]')
        .each((_, elem) => {
        const linkRoot = root(elem);
        const styleHref = linkRoot.attr('href');
        linkRoot.attr('href', util_1.removeTrailingSlash(baseUrl) + util_1.addLeadingSlash(styleHref));
        linkRoot.removeAttr('snowpack-baseurl');
    });
}
async function extractInlineScripts(htmlData) {
    const { file, root, getScripts, getStyles } = htmlData;
    getScripts().each((i, elem) => {
        const scriptRoot = root(elem);
        const scriptContent = scriptRoot.contents().text();
        if (!scriptContent) {
            return;
        }
        scriptRoot.empty();
        fs_1.writeFileSync(file + `.inline.${i}.js`, scriptContent);
        scriptRoot.attr('src', `./${path_1.default.basename(file)}.inline.${i}.js`);
        scriptRoot.attr('snowpack-inline', `true`);
    });
    getStyles().each((i, elem) => {
        const styleRoot = root(elem);
        const styleContent = styleRoot.contents().text();
        if (!styleContent) {
            return;
        }
        styleRoot.after(`<link rel="stylesheet" href="./${path_1.default.basename(file)}.inline.${i}.css" snowpack-inline="true" />`);
        styleRoot.remove();
        fs_1.writeFileSync(file + `.inline.${i}.css`, styleContent);
    });
}
async function restitchInlineScripts(htmlData) {
    const { file, root, getScripts, getLinks } = htmlData;
    getScripts()
        .filter('[snowpack-inline]')
        .each((_, elem) => {
        const scriptRoot = root(elem);
        const scriptFile = path_1.default.resolve(file, '..', scriptRoot.attr('src'));
        const scriptContent = fs_1.readFileSync(scriptFile, 'utf-8');
        scriptRoot.text(scriptContent);
        scriptRoot.removeAttr('src');
        scriptRoot.removeAttr('snowpack-inline');
        fs_1.unlinkSync(scriptFile);
    });
    getLinks('stylesheet')
        .filter('[snowpack-inline]')
        .each((_, elem) => {
        const linkRoot = root(elem);
        const styleFile = path_1.default.resolve(file, '..', linkRoot.attr('href'));
        const styleContent = fs_1.readFileSync(styleFile, 'utf-8');
        const newStyleEl = root('<style></style>');
        newStyleEl.text(styleContent);
        linkRoot.after(newStyleEl);
        linkRoot.remove();
        fs_1.unlinkSync(styleFile);
    });
}
/** Add new bundled CSS files to the HTML entrypoint file, if not already there. */
function addNewBundledCss(htmlData, manifest, baseUrl) {
    if (!manifest.outputs) {
        return;
    }
    for (const key of Object.keys(manifest.outputs)) {
        if (!key.endsWith('.css')) {
            continue;
        }
        const scriptKey = key.replace('.css', '.js');
        if (!manifest.outputs[scriptKey]) {
            continue;
        }
        const hasCssImportAlready = htmlData
            .getLinks('stylesheet')
            .toArray()
            .some((v) => v.attribs.href.includes(util_1.removeLeadingSlash(key)));
        const hasScriptImportAlready = htmlData
            .getScripts()
            .toArray()
            .some((v) => v.attribs.src.includes(util_1.removeLeadingSlash(scriptKey)));
        if (hasCssImportAlready || !hasScriptImportAlready) {
            continue;
        }
        const linkHref = util_1.removeTrailingSlash(baseUrl) + util_1.addLeadingSlash(key);
        htmlData.root('head').append(`<link rel="stylesheet" href="${linkHref}" />`);
    }
}
/**
 * Traverse the entrypoint for JS scripts, and add preload links to the HTML entrypoint.
 */
function preloadEntrypoint(htmlData, manifest, config) {
    const { root, getScripts } = htmlData;
    const preloadScripts = getScripts()
        .map((_, elem) => elem.attribs.src)
        .get()
        .filter(util_1.isTruthy);
    const collectedDeepImports = new Set();
    for (const preloadScript of preloadScripts) {
        const normalizedPreloadScript = util_1.removeLeadingSlash(path_1.default.posix.resolve('/', util_1.removeLeadingSlash(preloadScript)));
        collectDeepImports(normalizedPreloadScript, manifest, collectedDeepImports);
    }
    const baseUrl = config.buildOptions.baseUrl;
    for (const imp of collectedDeepImports) {
        const preloadUrl = (baseUrl ? util_1.removeTrailingSlash(baseUrl) : '') + util_1.addLeadingSlash(imp);
        root('head').append(`<link rel="modulepreload" href="${preloadUrl}" />`);
    }
}
/**
 * Handle the many different user input formats to return an array of strings.
 * resolve "auto" mode here.
 */
async function getEntrypoints(entrypoints, allBuildFiles) {
    if (entrypoints === 'auto') {
        // TODO: Filter allBuildFiles by HTML with head & body
        return allBuildFiles.filter((f) => f.endsWith('.html'));
    }
    if (Array.isArray(entrypoints)) {
        return entrypoints;
    }
    if (typeof entrypoints === 'function') {
        return entrypoints({ files: allBuildFiles });
    }
    throw new Error('UNEXPECTED ENTRYPOINTS: ' + entrypoints);
}
/**
 * Resolve an array of string entrypoints to absolute file paths. Handle
 * source vs. build directory relative entrypoints here as well.
 */
async function resolveEntrypoints(entrypoints, cwd, buildDirectoryLoc, config) {
    return Promise.all(entrypoints.map(async (entrypoint) => {
        if (path_1.default.isAbsolute(entrypoint)) {
            return entrypoint;
        }
        const buildEntrypoint = path_1.default.resolve(buildDirectoryLoc, entrypoint);
        if (await fs_1.promises.stat(buildEntrypoint).catch(() => null)) {
            return buildEntrypoint;
        }
        const resolvedSourceFile = path_1.default.resolve(cwd, entrypoint);
        let resolvedSourceEntrypoint;
        if (await fs_1.promises.stat(resolvedSourceFile).catch(() => null)) {
            const resolvedSourceUrl = file_urls_1.getUrlForFile(resolvedSourceFile, config);
            if (resolvedSourceUrl) {
                resolvedSourceEntrypoint = path_1.default.resolve(buildDirectoryLoc, util_1.removeLeadingSlash(resolvedSourceUrl));
                if (await fs_1.promises.stat(resolvedSourceEntrypoint).catch(() => null)) {
                    return resolvedSourceEntrypoint;
                }
            }
        }
        logger_1.logger.error(`Error: entrypoint "${entrypoint}" not found in either build or source:`, {
            name: 'optimize',
        });
        logger_1.logger.error(`  ✘ Build Entrypoint: ${buildEntrypoint}`, { name: 'optimize' });
        logger_1.logger.error(`  ✘ Source Entrypoint: ${resolvedSourceFile} ${resolvedSourceEntrypoint ? `-> ${resolvedSourceEntrypoint}` : ''}`, { name: 'optimize' });
        throw new Error(`Optimize entrypoint "${entrypoint}" does not exist.`);
    }));
}
/**
 * Process your entrypoints as either all JS or all HTML. If HTML,
 * scan those HTML files and add a Cheerio-powered root document
 * so that we can modify the HTML files as we go.
 */
async function processEntrypoints(originalEntrypointValue, entrypointFiles, buildDirectoryLoc, baseUrl) {
    // If entrypoints are JS:
    if (entrypointFiles.every((f) => f.endsWith('.js'))) {
        return { htmlEntrypoints: null, bundleEntrypoints: entrypointFiles };
    }
    // If entrypoints are HTML:
    if (entrypointFiles.every((f) => f.endsWith('.html'))) {
        const rawHtmlEntrypoints = await scanHtmlEntrypoints(entrypointFiles);
        const htmlEntrypoints = rawHtmlEntrypoints.filter(util_1.isTruthy);
        if (originalEntrypointValue !== 'auto' &&
            rawHtmlEntrypoints.length !== htmlEntrypoints.length) {
            throw new Error('INVALID HTML ENTRYPOINTS: ' + originalEntrypointValue);
        }
        htmlEntrypoints.forEach((val) => extractBaseUrl(val, baseUrl));
        htmlEntrypoints.forEach(extractInlineScripts);
        const bundleEntrypoints = Array.from(htmlEntrypoints.reduce((all, val) => {
            val.getLinks('stylesheet').each((_, elem) => {
                if (!elem.attribs.href || util_1.isRemoteUrl(elem.attribs.href)) {
                    return;
                }
                const resolvedCSS = elem.attribs.href[0] === '/'
                    ? path_1.default.resolve(buildDirectoryLoc, util_1.removeLeadingSlash(elem.attribs.href))
                    : path_1.default.resolve(val.file, '..', elem.attribs.href);
                all.add(resolvedCSS);
            });
            val.getScripts().each((_, elem) => {
                if (!elem.attribs.src || util_1.isRemoteUrl(elem.attribs.src)) {
                    return;
                }
                const resolvedJS = elem.attribs.src[0] === '/'
                    ? path_1.default.join(buildDirectoryLoc, util_1.removeLeadingSlash(elem.attribs.src))
                    : path_1.default.join(val.file, '..', elem.attribs.src);
                all.add(resolvedJS);
            });
            return all;
        }, new Set()));
        return { htmlEntrypoints, bundleEntrypoints };
    }
    // If entrypoints are mixed or neither, throw an error.
    throw new Error('MIXED ENTRYPOINTS: ' + entrypointFiles);
}
/**
 * Run esbuild on the build directory. This is run regardless of bundle=true or false,
 * since we use the generated manifest in either case.
 */
async function runEsbuildOnBuildDirectory(bundleEntrypoints, config, esbuildService) {
    var _a;
    const { outputFiles, warnings } = await esbuildService.build({
        entryPoints: bundleEntrypoints,
        outdir: FAKE_BUILD_DIRECTORY,
        outbase: config.buildOptions.out,
        write: false,
        bundle: true,
        splitting: true,
        format: 'esm',
        platform: 'browser',
        metafile: path_1.default.join(config.buildOptions.out, 'build-manifest.json'),
        publicPath: config.buildOptions.baseUrl,
        minify: config.experiments.optimize.minify,
        target: config.experiments.optimize.target,
    });
    const manifestFile = outputFiles.find((f) => f.path.endsWith('build-manifest.json'));
    const manifestContents = manifestFile.text;
    const manifest = JSON.parse(manifestContents);
    if (!outputFiles) {
        throw new Error('EMPTY BUILD');
    }
    if (warnings.length > 0) {
        console.warn(warnings);
    }
    outputFiles.forEach((f) => (f.path = f.path.replace(FAKE_BUILD_DIRECTORY_REGEX, util_1.addTrailingSlash(config.buildOptions.out))));
    if (!((_a = config.experiments.optimize) === null || _a === void 0 ? void 0 : _a.bundle)) {
        delete manifest.outputs;
    }
    else {
        Object.entries(manifest.outputs).forEach(([f, val]) => {
            const newKey = f.replace(FAKE_BUILD_DIRECTORY_REGEX, '/');
            manifest.outputs[newKey] = val;
            delete manifest.outputs[f];
        });
    }
    logger_1.logger.debug(`outputFiles: ${JSON.stringify(outputFiles)}`);
    logger_1.logger.debug(`manifest: ${JSON.stringify(manifest)}`);
    return { outputFiles, manifest };
}
/** The main optimize function: runs optimization on a build directory. */
async function runBuiltInOptimize(config) {
    const originalCwd = process.cwd();
    const buildDirectoryLoc = config.buildOptions.out;
    const options = config.experiments.optimize;
    if (!options) {
        return;
    }
    logger_1.logger.warn('(early preview: experiments.optimize is experimental, and still subject to change.)', {
        name: 'optimize',
    });
    // * Scan to collect all build files: We'll need this throughout.
    const allBuildFiles = glob_1.glob.sync('**/*', {
        cwd: buildDirectoryLoc,
        nodir: true,
        absolute: true,
    });
    // * Resolve and validate your entrypoints: they may be JS or HTML
    const userEntrypoints = await getEntrypoints(options.entrypoints, allBuildFiles);
    logger_1.logger.debug(JSON.stringify(userEntrypoints), { name: 'optimize.entrypoints' });
    const resolvedEntrypoints = await resolveEntrypoints(userEntrypoints, originalCwd, buildDirectoryLoc, config);
    logger_1.logger.debug('(resolved) ' + JSON.stringify(resolvedEntrypoints), { name: 'optimize.entrypoints' });
    const { htmlEntrypoints, bundleEntrypoints } = await processEntrypoints(options.entrypoints, resolvedEntrypoints, buildDirectoryLoc, config.buildOptions.baseUrl);
    logger_1.logger.debug(`htmlEntrypoints: ${JSON.stringify(htmlEntrypoints === null || htmlEntrypoints === void 0 ? void 0 : htmlEntrypoints.map((f) => f.file))}`);
    logger_1.logger.debug(`bundleEntrypoints: ${JSON.stringify(bundleEntrypoints)}`);
    if ((!htmlEntrypoints || htmlEntrypoints.length === 0) && bundleEntrypoints.length === 0) {
        throw new Error('[optimize] No HTML entrypoints detected. Set "entrypoints" manually if your site HTML is generated outside of Snowpack (SSR, Rails, PHP, etc.).');
    }
    // NOTE: esbuild has no `cwd` support, and assumes that you're always bundling the
    // current working directory. To get around this, we change the current working directory
    // for this run only, and then reset it on exit.
    process.chdir(buildDirectoryLoc);
    // * Run esbuild on the entire build directory. Even if you are not writing the result
    // to disk (bundle: false), we still use the bundle manifest as an in-memory representation
    // of our import graph, saved to disk.
    const esbuildService = await esbuild.startService();
    const { manifest, outputFiles } = await runEsbuildOnBuildDirectory(bundleEntrypoints, config, esbuildService);
    // * BUNDLE: TRUE - Save the bundle result to the build directory, and clean up to remove all original
    // build files that now live in the bundles.
    if (options.bundle) {
        for (const bundledInput of Object.keys(manifest.inputs)) {
            if (!manifest.outputs[bundledInput]) {
                logger_1.logger.debug(`Removing bundled source file: ${path_1.default.resolve(buildDirectoryLoc, bundledInput)}`);
                await fs_1.promises.unlink(path_1.default.resolve(buildDirectoryLoc, bundledInput));
            }
        }
        rimraf_1.default.sync(path_1.default.resolve(buildDirectoryLoc, util_1.removeLeadingSlash(config.buildOptions.webModulesUrl)));
        await removeEmptyFolders(buildDirectoryLoc);
        for (const outputFile of outputFiles) {
            mkdirp_1.default.sync(path_1.default.dirname(outputFile.path));
            await fs_1.promises.writeFile(outputFile.path, outputFile.contents);
        }
        if (htmlEntrypoints) {
            for (const htmlEntrypoint of htmlEntrypoints) {
                addNewBundledCss(htmlEntrypoint, manifest, config.buildOptions.baseUrl);
            }
        }
    }
    // * BUNDLE: FALSE - Just minifying & transform the CSS & JS files in place.
    else if (options.minify || options.target) {
        for (const f of allBuildFiles) {
            if (['.js', '.css'].includes(path_1.default.extname(f))) {
                let code = await fs_1.promises.readFile(f, 'utf-8');
                const minified = await esbuildService.transform(code, {
                    sourcefile: path_1.default.basename(f),
                    loader: path_1.default.extname(f).slice(1),
                    minify: options.minify,
                    target: options.target,
                });
                code = minified.code;
                await fs_1.promises.writeFile(f, code);
            }
        }
    }
    // * Restitch any inline scripts into HTML entrypoints that had been split out
    // for the sake of bundling/manifest.
    if (htmlEntrypoints) {
        for (const htmlEntrypoint of htmlEntrypoints) {
            restitchInlineScripts(htmlEntrypoint);
        }
    }
    // * PRELOAD: TRUE - Add preload link elements for each HTML entrypoint, to flatten
    // and optimize any deep import waterfalls.
    if (options.preload) {
        if (options.bundle) {
            throw new Error('preload is not needed when bundle=true, and cannot be used in combination.');
        }
        if (!htmlEntrypoints || htmlEntrypoints.length === 0) {
            throw new Error('preload only works with HTML entrypoints.');
        }
        for (const htmlEntrypoint of htmlEntrypoints) {
            preloadEntrypoint(htmlEntrypoint, manifest, config);
        }
    }
    // * Restitch any inline scripts into HTML entrypoints that had been split out
    // for the sake of bundling/manifest.
    if (htmlEntrypoints) {
        for (const htmlEntrypoint of htmlEntrypoints) {
            restitchBaseUrl(htmlEntrypoint, config.buildOptions.baseUrl);
        }
    }
    // Write the final HTML entrypoints to disk (if they exist).
    if (htmlEntrypoints) {
        for (const htmlEntrypoint of htmlEntrypoints) {
            await fs_1.promises.writeFile(htmlEntrypoint.file, htmlEntrypoint.root.html());
        }
    }
    // Write the final build manifest to disk.
    if (options.manifest) {
        await fs_1.promises.writeFile(path_1.default.join(config.buildOptions.out, 'build-manifest.json'), JSON.stringify(manifest));
    }
    // Cleanup and exit.
    esbuildService.stop();
    process.chdir(originalCwd);
    return;
}
exports.runBuiltInOptimize = runBuiltInOptimize;
//# sourceMappingURL=optimize.js.map