"use client"
import { useState, useEffect } from "react"
import { Button } from "@/components/ui/button"
import { Card } from "@/components/ui/card"
import { Badge } from "@/components/ui/badge"
import Link from "next/link"
import { ArrowLeft, ArrowRight, Eye, ExternalLink, Menu, X } from "lucide-react"
import Image from "next/image"

export default function ProjectsPage() {
  const [mounted, setMounted] = useState(false)
  const [activeFilter, setActiveFilter] = useState("all")
  const [isMenuOpen, setIsMenuOpen] = useState(false)

  const projects = [
    {
      id: 1,
      title: "AI-Powered E-commerce Platform",
      description:
        "Revolutionary online marketplace featuring intelligent product recommendations, dynamic pricing algorithms, and personalized shopping experiences that increased conversion rates by 340%.",
      image:
        "https://images.unsplash.com/photo-1460925895917-afdab827c52f?ixlib=rb-4.0.3&ixid=M3wxMjA3fDB8MHxwaG90by1wYWdlfHx8fGVufDB8fHx8fA%3D%3D&auto=format&fit=crop&w=2070&q=80",
      category: "ecommerce",
      technologies: ["React", "Node.js", "TensorFlow", "Stripe"],
      year: "2024",
      client: "TechCorp Inc.",
    },
    {
      id: 2,
      title: "Smart Banking Application",
      description:
        "Next-generation mobile banking platform with AI-driven fraud detection, biometric authentication, and real-time financial insights that serves over 100K+ active users.",
      image:
        "https://images.unsplash.com/photo-1551288049-bebda4e38f71?ixlib=rb-4.0.3&ixid=M3wxMjA3fDB8MHxwaG90by1wYWdlfHx8fGVufDB8fHx8fA%3D%3D&auto=format&fit=crop&w=2070&q=80",
      category: "applications",
      technologies: ["React Native", "Firebase", "Python", "AWS"],
      year: "2024",
      client: "FinanceFlow",
    },
    {
      id: 3,
      title: "Advanced E-commerce Solution",
      description:
        "Comprehensive multi-vendor marketplace with advanced inventory management, AI-powered customer insights, and automated marketing campaigns.",
      image:
        "https://images.unsplash.com/photo-1557804506-669a67965ba0?ixlib=rb-4.0.3&ixid=M3wxMjA3fDB8MHxwaG90by1wYWdlfHx8fGVufDB8fHx8fA%3D%3D&auto=format&fit=crop&w=2074&q=80",
      category: "ecommerce",
      technologies: ["Vue.js", "Laravel", "MySQL", "Redis"],
      year: "2023",
      client: "MarketPlace Pro",
    },
    {
      id: 4,
      title: "Smart Restaurant Platform",
      description:
        "Intelligent restaurant ecosystem with AI-powered menu optimization, dynamic pricing, real-time analytics, and seamless customer experience management.",
      image:
        "https://images.unsplash.com/photo-1555949963-ff9fe0c870eb?ixlib=rb-4.0.3&ixid=M3wxMjA3fDB8MHxwaG90by1wYWdlfHx8fGVufDB8fHx8fA%3D%3D&auto=format&fit=crop&w=2076&q=80",
      category: "websites",
      technologies: ["Next.js", "PostgreSQL", "OpenAI", "Vercel"],
      year: "2023",
      client: "RestaurantTech",
    },
    {
      id: 5,
      title: "Creative Agency Portfolio",
      description:
        "Stunning portfolio website for a creative agency featuring interactive animations, smooth transitions, and immersive user experience that showcases their creative work.",
      image:
        "https://images.unsplash.com/photo-1561070791-2526d30994b5?ixlib=rb-4.0.3&ixid=M3wxMjA3fDB8MHxwaG90by1wYWdlfHx8fGVufDB8fHx8fA%3D%3D&auto=format&fit=crop&w=2064&q=80",
      category: "websites",
      technologies: ["React", "Framer Motion", "GSAP", "Netlify"],
      year: "2024",
      client: "Creative Studio",
    },
    {
      id: 6,
      title: "Social Media Management App",
      description:
        "Comprehensive social media management platform with AI-powered content scheduling, analytics dashboard, and multi-platform posting capabilities.",
      image:
        "https://images.unsplash.com/photo-1611224923853-80b023f02d71?ixlib=rb-4.0.3&ixid=M3wxMjA3fDB8MHxwaG90by1wYWdlfHx8fGVufDB8fHx8fA%3D%3D&auto=format&fit=crop&w=2039&q=80",
      category: "applications",
      technologies: ["Vue.js", "Node.js", "MongoDB", "Socket.io"],
      year: "2023",
      client: "SocialTech",
    },
    {
      id: 7,
      title: "Brand Identity & Social Media Design",
      description:
        "Complete brand identity design and social media marketing campaign for a tech startup, including logo design, brand guidelines, and social media templates.",
      image:
        "https://images.unsplash.com/photo-1634942537034-2531766767d1?ixlib=rb-4.0.3&ixid=M3wxMjA3fDB8MHxwaG90by1wYWdlfHx8fGVufDB8fHx8fA%3D%3D&auto=format&fit=crop&w=2070&q=80",
      category: "social-media",
      technologies: ["Adobe Creative Suite", "Figma", "Canva", "After Effects"],
      year: "2024",
      client: "StartupTech",
    },
    {
      id: 8,
      title: "Luxury Fashion E-store",
      description:
        "High-end fashion e-commerce platform with premium user experience, advanced filtering, virtual try-on features, and seamless checkout process.",
      image:
        "https://images.unsplash.com/photo-1441986300917-64674bd600d8?ixlib=rb-4.0.3&ixid=M3wxMjA3fDB8MHxwaG90by1wYWdlfHx8fGVufDB8fHx8fA%3D%3D&auto=format&fit=crop&w=2070&q=80",
      category: "ecommerce",
      technologies: ["Shopify Plus", "React", "AR.js", "Stripe"],
      year: "2024",
      client: "Luxury Fashion",
    },
    {
      id: 9,
      title: "AI Content Generation Platform",
      description:
        "Revolutionary AI-powered content creation platform that generates high-quality articles, social media posts, and marketing copy using advanced natural language processing and machine learning algorithms.",
      image:
        "https://images.unsplash.com/photo-1677442136019-21780ecad995?ixlib=rb-4.0.3&ixid=M3wxMjA3fDB8MHxwaG90by1wYWdlfHx8fGVufDB8fHx8fA%3D%3D&auto=format&fit=crop&w=2070&q=80",
      category: "ai-projects",
      technologies: ["OpenAI GPT-4", "Python", "FastAPI", "React"],
      year: "2024",
      client: "ContentAI",
    },
    {
      id: 10,
      title: "Intelligent Document Analysis System",
      description:
        "Advanced AI system that automatically extracts, categorizes, and analyzes information from complex documents, reducing manual processing time by 85% and improving accuracy.",
      image:
        "https://images.unsplash.com/photo-1555949963-aa79dcee981c?ixlib=rb-4.0.3&ixid=M3wxMjA3fDB8MHxwaG90by1wYWdlfHx8fGVufDB8fHx8fA%3D%3D&auto=format&fit=crop&w=2070&q=80",
      category: "ai-projects",
      technologies: ["TensorFlow", "Computer Vision", "NLP", "Azure"],
      year: "2024",
      client: "DocuTech Solutions",
    },
    {
      id: 11,
      title: "AI-Powered Customer Support Bot",
      description:
        "Intelligent chatbot system with natural language understanding, sentiment analysis, and automated ticket routing that handles 90% of customer inquiries without human intervention.",
      image:
        "https://images.unsplash.com/photo-1531746790731-6c087fecd65a?ixlib=rb-4.0.3&ixid=M3wxMjA3fDB8MHxwaG90by1wYWdlfHx8fGVufDB8fHx8fA%3D%3D&auto=format&fit=crop&w=2006&q=80",
      category: "ai-projects",
      technologies: ["Dialogflow", "Node.js", "Machine Learning", "WebSocket"],
      year: "2024",
      client: "SupportAI Inc.",
    },
  ]

  const filteredProjects =
    activeFilter === "all"
      ? projects
      : activeFilter === "websites"
        ? projects.filter((project) => project.category === "websites" || project.category === "ecommerce")
        : projects.filter((project) => project.category === activeFilter)

  useEffect(() => {
    setMounted(true)
  }, [])

  if (!mounted) return null

  return (
    <div className="min-h-screen bg-white text-gray-900">
      {/* Header */}
      <header className="sticky top-0 z-50 backdrop-blur-3xl bg-[#161d25]/95 border-b border-white/10 shadow-2xl">
        <div className="container mx-auto px-4 py-6 relative">
          <div className="flex items-center justify-between">
            <Link href="/" className="flex items-center gap-4">
              <div className="relative group">
                <div className="absolute inset-0 bg-gradient-to-br from-gray-500/20 to-gray-400/10 rounded-3xl blur-xl group-hover:blur-2xl transition-all duration-700"></div>
                <Image
                  src="https://i.ibb.co/5XLC8VTC/kdsgnpng2.png"
                  alt="Kristijan Digital"
                  width={48}
                  height={48}
                  className="rounded-3xl shadow-2xl transition-transform duration-700 group-hover:scale-105 relative z-10 border border-white/10"
                />
                <div className="absolute inset-0 rounded-3xl bg-gradient-to-br from-white/10 to-transparent group-hover:from-white/20 transition-all duration-700"></div>
              </div>
              <div>
                <h1 className="text-2xl font-bold text-white">kristijan.digital</h1>
                <p className="text-xs text-white/60 font-medium">Digital Innovation Specialist</p>
              </div>
            </Link>

            {/* Desktop Navigation */}
            <nav className="hidden lg:flex items-center gap-12">
              {[
                { href: "/", label: "Home" },
                { href: "/#about", label: "About" },
                { href: "/#expertise", label: "Expertise" },
                { href: "/projects", label: "Projects" },
                { href: "/#testimonials", label: "Testimonials" },
              ].map((item) => (
                <Link
                  key={item.label}
                  href={item.href}
                  className="text-sm font-medium text-white/70 hover:text-white hover:scale-110 transition-all duration-500 relative group"
                >
                  {item.label}
                  <div className="absolute -bottom-2 left-0 w-0 h-0.5 bg-gradient-to-r from-gray-400 to-gray-300 group-hover:w-full transition-all duration-500"></div>
                </Link>
              ))}
            </nav>

            <div className="flex items-center gap-6">
              <Link href="/contact">
                <Button className="hidden lg:flex bg-gradient-to-r from-white/15 to-white/10 backdrop-blur-xl border border-white/30 hover:bg-gradient-to-r hover:from-white/25 hover:to-white/20 hover:border-white/50 shadow-xl hover:shadow-2xl hover:shadow-white/20 transition-all duration-500 hover:scale-105 rounded-xl px-8 py-3 font-semibold text-white">
                  Let's Work Together
                  <ArrowRight className="ml-2 h-4 w-4" />
                </Button>
              </Link>

              {/* Mobile Menu Button */}
              <Button
                variant="ghost"
                size="icon"
                className="lg:hidden rounded-2xl bg-white/5 backdrop-blur-xl border border-white/10 hover:bg-white/10 hover:border-white/30 transition-all duration-500"
                onClick={() => setIsMenuOpen(!isMenuOpen)}
              >
                {isMenuOpen ? <X className="h-5 w-5 text-white" /> : <Menu className="h-5 w-5 text-white" />}
              </Button>
            </div>
          </div>

          {/* Mobile Navigation */}
          {isMenuOpen && (
            <nav className="lg:hidden mt-6 pb-6 border-t border-white/10 pt-6 bg-black/20 backdrop-blur-3xl rounded-3xl border border-white/10 shadow-2xl">
              <div className="flex flex-col gap-6">
                {[
                  { href: "/", label: "Home" },
                  { href: "/#about", label: "About" },
                  { href: "/#expertise", label: "Expertise" },
                  { href: "/projects", label: "Projects" },
                  { href: "/#testimonials", label: "Testimonials" },
                ].map((item) => (
                  <Link
                    key={item.label}
                    href={item.href}
                    className="text-sm font-medium text-white/80 hover:text-white transition-colors duration-300 px-4"
                  >
                    {item.label}
                  </Link>
                ))}
                <Link href="/contact" className="px-4">
                  <Button className="w-full mt-4 bg-white/10 backdrop-blur-xl border border-white/20 hover:bg-white/20 rounded-2xl text-white">
                    Let's Collaborate
                  </Button>
                </Link>
              </div>
            </nav>
          )}
        </div>
      </header>

      {/* Hero Section */}
      <section className="relative py-32 overflow-hidden bg-[#161d25]">
        <div className="absolute inset-0 bg-[radial-gradient(circle_at_20%_80%,rgba(255,255,255,0.1),transparent_70%)]"></div>
        <div className="absolute inset-0 bg-[radial-gradient(circle_at_80%_20%,rgba(255,255,255,0.08),transparent_70%)]"></div>

        <div className="container mx-auto px-4 relative z-10">
          <div className="text-center mb-12">
            <h1 className="text-4xl md:text-6xl font-bold mb-6 text-white leading-tight">All Projects</h1>
            <p className="text-xl text-white/70 max-w-3xl mx-auto mb-8 leading-relaxed">
              Explore my complete portfolio of innovative digital solutions that have transformed businesses and
              exceeded expectations
            </p>
          </div>
        </div>
      </section>

      {/* Projects Section */}
      <section className="py-32 bg-white">
        <div className="container mx-auto px-4">
          {/* Project Filters */}
          <div className="flex flex-wrap justify-center gap-4 mb-16">
            {[
              { key: "all", label: "All Works" },
              { key: "websites", label: "Websites" },
              { key: "applications", label: "Applications" },
              { key: "social-media", label: "Social Media Design" },
              { key: "ai-projects", label: "AI Projects" },
            ].map((filter) => (
              <Button
                key={filter.key}
                variant={activeFilter === filter.key ? "default" : "outline"}
                onClick={() => setActiveFilter(filter.key)}
                className={`rounded-full px-8 py-3 font-semibold transition-all duration-500 hover:scale-105 ${
                  activeFilter === filter.key
                    ? "bg-black text-white shadow-2xl hover:bg-gray-800"
                    : "bg-white border-2 border-gray-200 hover:border-gray-400 hover:bg-gray-50 text-black hover:shadow-xl"
                }`}
              >
                {filter.label}
              </Button>
            ))}
          </div>

          {/* Projects Grid */}
          <div className="grid md:grid-cols-2 lg:grid-cols-3 gap-8">
            {filteredProjects.map((project, index) => (
              <Card
                key={project.id}
                className="group overflow-hidden bg-white border-2 border-gray-200 shadow-xl hover:shadow-2xl hover:shadow-gray-300/50 transition-all duration-700 hover:scale-105 hover:-translate-y-4 rounded-3xl cursor-pointer h-full flex flex-col"
              >
                <div className="relative overflow-hidden">
                  <Image
                    src={project.image || "/placeholder.svg"}
                    alt={project.title}
                    width={600}
                    height={400}
                    className="w-full h-64 object-cover transition-transform duration-700 group-hover:scale-110"
                  />
                  <div className="absolute inset-0 bg-black/0 group-hover:bg-black/20 transition-all duration-500"></div>
                  <div className="absolute inset-0 flex items-center justify-center opacity-0 group-hover:opacity-100 transition-all duration-500">
                    <div className="flex gap-4">
                      <Link href={`/projects/${project.id}`}>
                        <Button
                          size="icon"
                          variant="secondary"
                          className="rounded-2xl bg-white/95 backdrop-blur-3xl border border-white/30 hover:bg-white hover:scale-125 hover:shadow-2xl transition-all duration-500 w-14 h-14"
                        >
                          <Eye className="h-6 w-6 text-black" />
                        </Button>
                      </Link>
                      <Button
                        size="icon"
                        variant="secondary"
                        className="rounded-2xl bg-white/95 backdrop-blur-3xl border border-white/30 hover:bg-white hover:scale-125 hover:shadow-2xl transition-all duration-500 w-14 h-14"
                        onClick={() => window.open('https://kristijan.digital', '_blank')}
                      >
                        <ExternalLink className="h-6 w-6 text-black" />
                      </Button>
                    </div>
                  </div>

                  {/* Project Year Badge */}
                  <div className="absolute top-6 right-6 bg-black/90 backdrop-blur-2xl px-4 py-2 rounded-full border border-white/20 text-sm font-medium text-white">
                    {project.year}
                  </div>
                </div>

                <div className="p-8 flex-1 flex flex-col">
                  <div className="flex items-center gap-2 mb-4">
                    <span className="text-xs font-semibold text-gray-500 bg-gray-100 px-3 py-1 rounded-full">
                      {project.year}
                    </span>
                    <span className="text-xs font-semibold text-gray-500 bg-gray-100 px-3 py-1 rounded-full">
                      {project.client}
                    </span>
                  </div>

                  <h3 className="text-xl font-bold text-black mb-3 group-hover:text-gray-700 transition-colors duration-500">
                    {project.title}
                  </h3>

                  <p className="text-gray-600 leading-relaxed mb-4 text-sm flex-1">{project.description}</p>

                  <div className="flex flex-wrap gap-2 mb-6 mt-auto">
                    {project.technologies.slice(0, 3).map((tech, techIndex) => (
                      <Badge
                        key={techIndex}
                        variant="secondary"
                        className="text-xs bg-gray-100 border border-gray-200 hover:bg-gray-200 hover:border-gray-300 transition-colors duration-500 rounded-xl px-3 py-1 text-gray-700"
                      >
                        {tech}
                      </Badge>
                    ))}
                    {project.technologies.length > 3 && (
                      <Badge
                        variant="secondary"
                        className="text-xs bg-gray-50 border border-gray-200 rounded-xl px-3 py-1 text-gray-500"
                      >
                        +{project.technologies.length - 3} more
                      </Badge>
                    )}
                  </div>

                  <Link href={`/projects/${project.id}`}>
                    <Button
                      variant="ghost"
                      className="group/btn p-0 h-auto hover:text-gray-600 transition-colors duration-500 text-gray-700 font-semibold"
                    >
                      View Case Study
                      <ArrowRight className="ml-2 h-4 w-4 group-hover/btn:translate-x-2 transition-transform duration-500" />
                    </Button>
                  </Link>
                </div>
              </Card>
            ))}
          </div>
        </div>
      </section>

      {/* Ready to Start Section + Footer Combined */}
      <section className="relative py-20 overflow-hidden bg-[#161d25]">
        <div className="absolute inset-0 overflow-hidden">
          <div className="absolute top-[25%] left-[30%] w-2 h-2 bg-white/20 rounded-full animate-float"></div>
          <div className="absolute top-[45%] right-[40%] w-1 h-1 bg-white/30 rounded-full animate-float-delayed"></div>
          <div className="absolute bottom-[35%] left-[45%] w-3 h-3 bg-white/15 rounded-full animate-float-slow"></div>
          <div className="absolute top-[75%] right-[25%] w-1.5 h-1.5 bg-white/25 rounded-full animate-float"></div>
          <div className="absolute bottom-[55%] right-[75%] w-2 h-2 bg-white/18 rounded-full animate-float-delayed"></div>
          <div className="absolute top-[65%] left-[15%] w-1 h-1 bg-white/22 rounded-full animate-float-slow"></div>
        </div>

        <div className="container mx-auto px-4 relative z-10">
          {/* Ready Badge */}
          <div className="text-center mb-12">
            <div className="inline-flex items-center gap-2 bg-white/10 backdrop-blur-xl border border-white/20 rounded-full px-4 py-2 mb-8">
              <div className="w-2 h-2 bg-green-400 rounded-full animate-pulse"></div>
              <span className="text-white/90 text-sm font-medium">Ready to start your project?</span>
            </div>
          </div>

          {/* Main Content */}
          <div className="text-center mb-16">
            <h2 className="text-4xl md:text-6xl font-bold text-white mb-6 leading-tight text-balance">
              Let's Create Something Extraordinary
            </h2>
            <p className="text-xl text-white/70 max-w-3xl mx-auto leading-relaxed mb-12 text-pretty">
              Transform your digital vision into reality with innovative solutions that drive growth, engage users, and
              deliver exceptional results for your business.
            </p>
            <Link href="/contact">
              <Button
                size="lg"
                className="bg-white text-black hover:bg-white/90 rounded-full px-8 py-4 text-lg font-semibold transition-all duration-300 hover:scale-105 hover:shadow-xl"
              >
                Start Your Project
                <ArrowRight className="ml-3 h-5 w-5" />
              </Button>
            </Link>
          </div>

          {/* Social Icons */}
          <div className="flex items-center justify-center gap-6 mb-16">
            <a
              href="https://linkedin.com/in/kristijan-digital"
              target="_blank"
              rel="noopener noreferrer"
              className="w-12 h-12 bg-white/10 backdrop-blur-xl border border-white/20 rounded-full flex items-center justify-center text-white/70 hover:text-white hover:bg-white/20 transition-all duration-300 hover:scale-110"
            >
              <svg className="w-5 h-5" fill="currentColor" viewBox="0 0 24 24">
                <path d="M20.447 20.452h-3.554v-5.569c0-1.328-.027-3.037-1.852-3.037-1.853 0-2.136 1.445-2.136 2.939v5.667H9.351V9h3.414v1.561h.046c.477-.9 1.637-1.85 3.37-1.85 3.601 0 4.267 2.37 4.267 5.455v6.286zM5.337 7.433c-1.144 0-2.063-.926-2.063-2.065 0-1.138.92-2.063 2.063-2.063 1.14 0 2.064.925 2.064 2.063 0 1.139-.925 2.065-2.064 2.065zm1.782 13.019H3.555V9h3.564v11.452zM22.225 0H1.771C.792 0 0 .774 0 1.729v20.542C0 23.227.792 24 1.771 24h20.451C23.2 24 24 23.227 24 22.271V1.729C24 .774 23.2 0 22.222 0h.003z" />
              </svg>
            </a>
            <a
              href="https://github.com/kristijan-digital"
              target="_blank"
              rel="noopener noreferrer"
              className="w-12 h-12 bg-white/10 backdrop-blur-xl border border-white/20 rounded-full flex items-center justify-center text-white/70 hover:text-white hover:bg-white/20 transition-all duration-300 hover:scale-110"
            >
              <svg className="w-5 h-5" fill="currentColor" viewBox="0 0 24 24">
                <path d="M20.447 20.452h-3.554v-5.569c0-1.328-.027-3.037-1.852-3.037-1.853 0-2.136 1.445-2.136 2.939v5.667H9.351V9h3.414v1.561h.046c.477-.9 1.637-1.85 3.37-1.85 3.601 0 4.267 2.37 4.267 5.455v6.286zM5.337 7.433c-1.144 0-2.063-.926-2.063-2.065 0-1.138.92-2.063 2.063-2.063 1.14 0 2.064.925 2.064 2.063 0 1.139-.925 2.065-2.064 2.065zm1.782 13.019H3.555V9h3.564v11.452zM22.225 0H1.771C.792 0 0 .774 0 1.729v20.542C0 23.227.792 24 1.771 24h20.451C23.2 24 24 23.227 24 22.271V1.729C24 .774 23.2 0 22.222 0h.003z" />
              </svg>
            </a>
            <a
              href="https://instagram.com/kristijan.digital"
              target="_blank"
              rel="noopener noreferrer"
              className="w-12 h-12 bg-white/10 backdrop-blur-xl border border-white/20 rounded-full flex items-center justify-center text-white/70 hover:text-white hover:bg-white/20 transition-all duration-300 hover:scale-110"
            >
              <svg className="w-5 h-5" fill="currentColor" viewBox="0 0 24 24">
                <path d="M12 2.163c3.204 0 3.584.012 4.85.07 3.252.148 4.771 1.691 4.919 4.919.058 1.265.069 1.645.069 4.849 0 3.205-.012 3.584-.069 4.849-.149 3.225-1.664 4.771-4.919 4.919-1.266.058-1.644.07-4.85.07-3.204 0-3.584-.012-4.849-.07-3.26-.149-4.771-1.699-4.919-4.92-.058-1.265-.07-1.644-.07-4.849 0-3.204.013-3.583.07-4.849.149-3.227 1.664-4.771 4.919-4.919 1.266-.057 1.645-.069 4.849-.069zm0-2.163c-3.259 0-3.667.014-4.947.072-4.358.2-6.78 2.618-6.98 6.98-.059 1.281-.073 1.689-.073 4.948 0 3.259.014 3.668.072 4.948.2 4.358 2.618 6.78 6.98 6.98 1.281.058 1.689.072 4.948.072 3.259 0 3.668-.014 4.948-.072 4.354-.2 6.782-2.618 6.979-6.98.059-1.28.073-1.689.073-4.948 0-3.259-.014-3.667-.072-4.947-.196-4.354-2.617-6.78-6.979-6.98-1.281-.059-1.69-.073-4.949-.073zm0 5.838c-3.403 0-6.162 2.759-6.162 6.162s2.759 6.163 6.162 6.163 6.162-2.759 6.162-6.163c0-3.403-2.759-6.162-6.162-6.162zm0 10.162c-2.209 0-4-1.79-4-4 0-2.209 1.791-4 4-4s4 1.791 4 4c0 2.21-1.791 4-4 4zm6.406-11.845c-.796 0-1.441.645-1.441 1.44s.645 1.44 1.441 1.44c.795 0 1.439-.645 1.439-1.44s-.644-1.44-1.439-1.44z" />
              </svg>
            </a>
            <a
              href="mailto:hello@kristijan.digital"
              className="w-12 h-12 bg-white/10 backdrop-blur-xl border border-white/20 rounded-full flex items-center justify-center text-white/70 hover:text-white hover:bg-white/20 transition-all duration-300 hover:scale-110"
            >
              <svg className="w-5 h-5" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                <path
                  strokeLinecap="round"
                  strokeLinejoin="round"
                  strokeWidth={2}
                  d="M3 8l7.89 4.26a2 2 0 002.22 0L21 8M5 19h14a2 2 0 002-2V7a2 2 0 00-2-2H5a2 2 0 00-2 2v10a2 2 0 002 2z"
                />
              </svg>
            </a>
          </div>

          {/* Copyright */}
          <div className="text-center">
            <p className="text-white/60 text-sm">© 2024 Kristijan Design. All rights reserved.</p>
            <p className="text-white/60 text-sm mt-2 flex items-center justify-center gap-2">
              Crafted with <span className="text-white animate-pulse">❤️</span> and innovation
            </p>
          </div>
        </div>
      </section>
    </div>
  )
}
