"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.createImportResolver = exports.getImportStats = void 0;
const fs_1 = __importDefault(require("fs"));
const path_1 = __importDefault(require("path"));
const util_1 = require("../util");
const file_urls_1 = require("./file-urls");
const cwd = process.cwd();
/** Perform a file disk lookup for the requested import specifier. */
function getImportStats(importedFileOnDisk) {
    try {
        return fs_1.default.statSync(importedFileOnDisk);
    }
    catch (err) {
        // file doesn't exist, that's fine
    }
    return false;
}
exports.getImportStats = getImportStats;
/** Resolve an import based on the state of the file/folder found on disk. */
function resolveSourceSpecifier(spec, stats, config) {
    // Handle directory imports (ex: "./components" -> "./components/index.js")
    if (stats && stats.isDirectory()) {
        const trailingSlash = spec.endsWith('/') ? '' : '/';
        spec = spec + trailingSlash + 'index.js';
    }
    // Transform the file extension (from input to output)
    const { baseExt } = util_1.getExt(spec);
    const extToReplace = config._extensionMap[baseExt];
    if (extToReplace) {
        spec = util_1.replaceExt(spec, baseExt, extToReplace);
    }
    // Lazy check to handle imports that are missing file extensions
    if (!stats && !spec.endsWith('.js') && !spec.endsWith('.css')) {
        spec = spec + '.js';
    }
    return spec;
}
/**
 * Create a import resolver function, which converts any import relative to the given file at "fileLoc"
 * to a proper URL. Returns false if no matching import was found, which usually indicates a package
 * not found in the import map.
 */
function createImportResolver({ fileLoc, config }) {
    return function importResolver(spec) {
        var _a;
        // Ignore "http://*" imports
        if (util_1.isRemoteUrl(spec)) {
            return spec;
        }
        // Ignore packages marked as external
        if ((_a = config.installOptions.externalPackage) === null || _a === void 0 ? void 0 : _a.includes(spec)) {
            return spec;
        }
        if (spec.startsWith('/')) {
            const importStats = getImportStats(path_1.default.resolve(cwd, spec.substr(1)));
            return resolveSourceSpecifier(spec, importStats, config);
        }
        if (spec.startsWith('./') || spec.startsWith('../')) {
            const importedFileLoc = path_1.default.resolve(path_1.default.dirname(fileLoc), spec);
            const importStats = getImportStats(importedFileLoc);
            const newSpec = file_urls_1.getUrlForFile(importedFileLoc, config) || spec;
            return resolveSourceSpecifier(newSpec, importStats, config);
        }
        const aliasEntry = util_1.findMatchingAliasEntry(config, spec);
        if (aliasEntry && (aliasEntry.type === 'path' || aliasEntry.type === 'url')) {
            const { from, to } = aliasEntry;
            let result = spec.replace(from, to);
            if (aliasEntry.type === 'url') {
                return result;
            }
            const importedFileLoc = path_1.default.resolve(cwd, result);
            const importStats = getImportStats(importedFileLoc);
            const newSpec = file_urls_1.getUrlForFile(importedFileLoc, config) || spec;
            return resolveSourceSpecifier(newSpec, importStats, config);
        }
        return false;
    };
}
exports.createImportResolver = createImportResolver;
//# sourceMappingURL=import-resolver.js.map